using System;
using System.Data;
using Microsoft.Reporting.WinForms;
using System.Reflection;
using System.IO;
using gov.va.med.vbecs.DAL.VAL;
using BROKEN = gov.va.med.vbecs.Common.VbecsBrokenRules.InappropriateTransfusionRequestReport;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;


namespace gov.va.med.vbecs.BOL
{

		#region Header

		///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
		///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
		///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
		///<Developers>
		///	<Developer>Unknown</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/28/2004</CreationDate>
		///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
		///<summary>Summary description for AuditTrail.</summary>

		#endregion

    public class AuditTrail : BOL.VbecsReport
    {
        #region Variable

        private DataTable _dtAuditReport;

        /// <summary>
        /// This class contins all string constants for audit trail
        /// </summary>
        public abstract class AuditTrailConst
        {
            /// <summary>
            /// String constants for audit identifier
            /// </summary>
            public abstract class AuditIdentifier
            {
                /// <summary>
                /// Constants
                /// </summary>
                public const string QCSetup = "Daily QC Setup Changed";
            }

            /// <summary>
            /// String constants for audit trail type
            /// </summary>
            public abstract class AuditTrailType
            {
                /// <summary>
                /// Constants
                /// </summary>
                public const string Patient = "Patient Changes",
                    TransfusionRequirement = "Patient Transfusion Requirements",
                    BloodUnitFinance = "Blood Unit Financial Changes",
                    LocalSupplier = "Maintenance: Local Facilities",
                    BloodProduct = "Maintenance: Blood Product",
                    MaintainDivision = "Maintenance: Division Configuration",
                    ProductModifications = "Maintenance: Modifications Configuration",
                    Antibody = "Maintenance: Antibody Configuration",
                    Workload = "Maintenance: Workload Configuration",
                    MSBOS = "Maintenance: MSBOS Configuration",
                    ComponentClass = "Maintenance: Component Class",
                    QCConfiguration = "Maintenance: Daily QC Configuration",
                    TestingConfiguration = "Maintenance: Testing Configuration",
                    ReagentInventoryLevel = "Maintenance: Minimum Reagent Inventory Levels",
                    UserAlerts = "Maintenance: User Alert Configuration",
                    BloodUnit = "Blood Unit Changes",
                    MaintainUser = "Maintenance: User Role Changes",
                    InterfaceControl = "Maintenance: Interface Control",
                    TransfusionEffectiveness = "Maintenance: Transfusion Effectiveness Changes",
                    TransfusionComplications = "Maintenance: Transfusion Complications Changes",
                    SpecimenTest = "Patient Testing Changes";
            }
        }

        #endregion

        #region Constructors

        ///<Developers>
        ///	<Developer>Krzysztof Dobranowski</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>3/24/2004</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="4001"> 
        ///		<ExpectedInput>Valid Start and End DateTime</ExpectedInput>
        ///		<ExpectedOutput>AuditTrail object initialized</ExpectedOutput>
        ///	</Case>
        ///
        ///<Case type="1" testid ="5062"> 
        ///		<ExpectedInput>NA</ExpectedInput>
        ///		<ExpectedOutput>NA</ExpectedOutput>
        ///	</Case>
        ///
        ///</TestCases>
        ///
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// Constructor
        /// </summary>
        /// <param name="startDate"></param>
        /// <param name="endDate"></param>
        public AuditTrail(DateTime startDate, DateTime endDate)
        {
            this.StartDate = startDate;
            this.EndDate = endDate;
            this._dtAuditReport = this.CreateAuditReportTable();
            SetInitialBrokenRules(BROKEN.FormClassName, BROKEN.RuleSets.BaseRuleSet);
        }

        ///<Developers>
        ///	<Developer>Rob Heiberger</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>1/31/2005</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="6644"> 
        ///		<ExpectedInput>Valid Parameters</ExpectedInput>
        ///		<ExpectedOutput>NA</ExpectedOutput>
        ///	</Case>
        ///
        ///
        ///<Case type="1" testid ="6645"> 
        ///		<ExpectedInput>Invalid Parameters</ExpectedInput>
        ///		<ExpectedOutput>NA</ExpectedOutput>
        ///	</Case>
        ///
        ///
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        /// <summary>
        /// AuditTrail
        /// </summary>
        public AuditTrail()
        {
            this._dtAuditReport = this.CreateAuditReportTable();
            SetInitialBrokenRules(BROKEN.FormClassName, BROKEN.RuleSets.BaseRuleSet);
        }
        #endregion

        #region Properties

        ///<Developers>
        ///	<Developer>Unknown</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>7/28/2004</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="5060"> 
        ///		<ExpectedInput>None</ExpectedInput>
        ///		<ExpectedOutput>AuditTrail datatable</ExpectedOutput>
        ///	</Case>
        ///
        ///<Case type="1" testid ="5061"> 
        ///		<ExpectedInput>NA</ExpectedInput>
        ///		<ExpectedOutput>NA</ExpectedOutput>
        ///	</Case>
        ///
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// DataTable containing audit report data
        /// </summary>
        public DataTable AuditReportData
        {
            get
            {
                return this._dtAuditReport;
            }
        }

        #endregion

        #region Methods

        /// <summary>
        /// Generates the DataTable used as a source for PatientAudit Report
        /// </summary>
        /// <returns></returns>
        private void GetPatientAuditReport()
        {
            DataTable dtPatientAudit = DAL.AuditTrail.GetPatientAudit(this.StartDate, this.EndDate);
            bool RowModified = false; //indicates if the audited row has been modified or newly added
            string PatientSsn = "";

            //generating one audit report row for each changed field
            int nCount = dtPatientAudit.Rows.Count;
            for (int i = 0; i < nCount; i++)
            {
                if ((DateTime)dtPatientAudit.Rows[i][ARTIFICIAL.MirrorDate] < this.StartDate)
                    continue;
                if (i == dtPatientAudit.Rows.Count - 1)
                {
                    RowModified = false;
                }
                else
                {
                    //do not compare records of two different patients
                    //if the record has just been added and there is no history of changes
                    if (dtPatientAudit.Rows[i][TABLE.Patient.PatientGuid].ToString() != dtPatientAudit.Rows[i + 1][TABLE.Patient.PatientGuid].ToString())
                    {
                        RowModified = false;
                    }
                    else
                    {
                        RowModified = true;
                    }
                }

                foreach (DataColumn dtCol in dtPatientAudit.Columns)
                {
                    if (dtCol.ColumnName.ToUpper() != ARTIFICIAL.Source &&
                        dtCol.ColumnName != ARTIFICIAL.MirrorId &&
                        dtCol.ColumnName != ARTIFICIAL.MirrorDate &&
                        dtCol.ColumnName != TABLE.Patient.LastUpdateDate &&
                        dtCol.ColumnName != TABLE.Patient.LastUpdateUser &&
                        dtCol.ColumnName != TABLE.Patient.LastUpdateFunctionId &&
                        dtCol.ColumnName != TABLE.Patient.PatientGuid)
                    {
                        //skip columns that have not changed the value
                        if (RowModified == true)
                        {
                            if (dtPatientAudit.Rows[i][dtCol].Equals(dtPatientAudit.Rows[i + 1][dtCol]))
                            {
                                continue;
                            }
                        }

                        DataRow drAuditReport = this._dtAuditReport.NewRow();
                        drAuditReport[ARTIFICIAL.AuditType] = AuditTrailConst.AuditTrailType.Patient;
                        PatientSsn = dtPatientAudit.Rows[i][TABLE.Patient.PatientSsn].ToString();
                        PatientSsn = PatientSsn.Substring(0, 3) + "-" + PatientSsn.Substring(3, 2) + "-" + PatientSsn.Substring(5);
                        drAuditReport[ARTIFICIAL.AuditIdentifier] = dtPatientAudit.Rows[i][TABLE.Patient.PatientLastName].ToString().Trim() + ", " +
                            dtPatientAudit.Rows[i][TABLE.Patient.PatientFirstName].ToString().Trim() + " " +
                            PatientSsn;
                        drAuditReport[ARTIFICIAL.ChangedFieldName] = Common.Utility.ParseStringwithSpaces(dtCol.ColumnName);
                        if (dtCol.DataType.ToString() == "System.DateTime")
                        {
                            if (RowModified == false)
                            {
                                drAuditReport[ARTIFICIAL.OldValue] = "None";
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.OldValue] = Common.VBECSDateTime.FormatDateString((DateTime)dtPatientAudit.Rows[i + 1][dtCol]);
                            }

                            drAuditReport[ARTIFICIAL.NewValue] = Common.VBECSDateTime.FormatDateString((DateTime)dtPatientAudit.Rows[i][dtCol]);
                        }
                        else
                        {
                            if (dtCol.ColumnName == TABLE.Patient.PatientSsn)
                            {
                                if (RowModified == false)
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "None";
                                }
                                else
                                {
                                    PatientSsn = dtPatientAudit.Rows[i + 1][dtCol].ToString();
                                    drAuditReport[ARTIFICIAL.OldValue] = PatientSsn.Substring(0, 3) + "-" + PatientSsn.Substring(3, 2) + "-" + PatientSsn.Substring(5);
                                }

                                PatientSsn = dtPatientAudit.Rows[i][dtCol].ToString();
                                drAuditReport[ARTIFICIAL.NewValue] = PatientSsn.Substring(0, 3) + "-" + PatientSsn.Substring(3, 2) + "-" + PatientSsn.Substring(5);
                            }
                            else
                            {
                                if (RowModified == false)
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "None";
                                }
                                else
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = dtPatientAudit.Rows[i + 1][dtCol];
                                }

                                drAuditReport[ARTIFICIAL.NewValue] = dtPatientAudit.Rows[i][dtCol];
                            }
                        }

                        try
                        {
                            BOL.VbecsUser LastUpdateUser = new BOL.VbecsUser(dtPatientAudit.Rows[i][TABLE.SpecimenTestThreshold.LastUpdateUser].ToString().Trim());
                            if (LastUpdateUser.UserName == null)
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = dtPatientAudit.Rows[i][TABLE.SpecimenTestThreshold.LastUpdateUser];
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = LastUpdateUser.UserName;
                            }
                        }
                        catch
                        {
                            drAuditReport[ARTIFICIAL.AuditUser] = dtPatientAudit.Rows[i][TABLE.SpecimenTestThreshold.LastUpdateUser];
                        }

                        drAuditReport[ARTIFICIAL.AuditDate] = dtPatientAudit.Rows[i][ARTIFICIAL.MirrorDate];
                        drAuditReport[ARTIFICIAL.Comment] = "";

                        this._dtAuditReport.Rows.Add(drAuditReport);
                    }
                }
            }
        }

        /// <summary>
        /// Generates the DataTable used as a source for Transfusion Requirement Audit Report
        /// </summary>
        private void GetTRAuditReport()
        {
            DataTable dtTRAudit = DAL.AuditTrail.GetTRAudit(this.StartDate, this.EndDate);
            string Comment = "";
            bool RowModified = false; //indicates if the audited row has been modified or newly added
            int nCount = 0;

            //generating one audit report row for each changed field
            nCount = dtTRAudit.Rows.Count;
            for (int i = 0; i < nCount; i++)
            {
                if ((DateTime)dtTRAudit.Rows[i][ARTIFICIAL.MirrorDate] < this.StartDate)
                    continue;
                if (i == dtTRAudit.Rows.Count - 1)
                {
                    RowModified = false;
                }
                else
                {
                    //check for newly added TRs
                    if (dtTRAudit.Rows[i][TABLE.PatientTransfusionRequirement.PatientTransfusionRequirementGuid].ToString() != dtTRAudit.Rows[i + 1][TABLE.PatientTransfusionRequirement.PatientTransfusionRequirementGuid].ToString())
                    {
                        RowModified = false;
                    }
                    //checking for inactivated TRs
                    else if (dtTRAudit.Rows[i][TABLE.PatientTransfusionRequirement.RecordStatusCode].ToString() == System.Convert.ToString(Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Inactive)) &&
                        dtTRAudit.Rows[i + 1][TABLE.PatientTransfusionRequirement.RecordStatusCode].ToString() == System.Convert.ToString(Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active)))
                    {
                        RowModified = true;
                    }
                    else
                    {
                        continue;
                    }
                }

                DataRow drAuditReport = this._dtAuditReport.NewRow();

                drAuditReport[ARTIFICIAL.AuditType] = AuditTrailConst.AuditTrailType.TransfusionRequirement;
                drAuditReport[ARTIFICIAL.AuditIdentifier] = dtTRAudit.Rows[i][TABLE.Patient.PatientLastName].ToString().Trim() + ", " +
                    dtTRAudit.Rows[i][TABLE.Patient.PatientFirstName].ToString().Trim() + " (" +
                    dtTRAudit.Rows[i][TABLE.Patient.DisplayVistaPatientId].ToString().Trim() + ")";
                drAuditReport[ARTIFICIAL.ChangedFieldName] = "Status Indicator for " + dtTRAudit.Rows[i][TABLE.TransfusionRequirementCategory.TransfusionRequirementCategoryText].ToString() + "(" + dtTRAudit.Rows[i][TABLE.PatientTransfusionRequirement.TransfusionRequirementText].ToString() + ")";

                if (RowModified == false)
                {
                    drAuditReport[ARTIFICIAL.OldValue] = "None";
                    drAuditReport[ARTIFICIAL.NewValue] = "Active";
                }
                else
                {
                    if (dtTRAudit.Rows[i][TABLE.PatientTransfusionRequirement.RecordStatusCode].ToString().Trim() == Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active).ToString())
                    {
                        drAuditReport[ARTIFICIAL.OldValue] = "Inactive";
                        drAuditReport[ARTIFICIAL.NewValue] = "Active";
                    }
                    else
                    {
                        drAuditReport[ARTIFICIAL.OldValue] = "Active";
                        drAuditReport[ARTIFICIAL.NewValue] = "Inactive";
                    }
                }


                try
                {
                    BOL.VbecsUser LastUpdateUser = new BOL.VbecsUser(dtTRAudit.Rows[i][TABLE.PatientTransfusionRequirement.LastUpdateUser].ToString().Trim());
                    if (LastUpdateUser.UserName == null)
                    {
                        drAuditReport[ARTIFICIAL.AuditUser] = dtTRAudit.Rows[i][TABLE.PatientTransfusionRequirement.LastUpdateUser];
                    }
                    else
                    {
                        drAuditReport[ARTIFICIAL.AuditUser] = LastUpdateUser.UserName;
                    }
                }
                catch
                {
                    drAuditReport[ARTIFICIAL.AuditUser] = dtTRAudit.Rows[i][TABLE.PatientTransfusionRequirement.LastUpdateUser];
                }

                drAuditReport[ARTIFICIAL.AuditDate] = dtTRAudit.Rows[i][ARTIFICIAL.MirrorDate];

                if (dtTRAudit.Rows[i].IsNull(TABLE.PatientTransfusionRequirement.InactivationComment) == false)
                {
                    Comment += dtTRAudit.Rows[i][TABLE.PatientTransfusionRequirement.InactivationComment].ToString().Trim();
                }

                drAuditReport[ARTIFICIAL.Comment] = Comment.Trim();
                Comment = "";
                this._dtAuditReport.Rows.Add(drAuditReport);
            }

            DataTable dtSIAudit = DAL.AuditTrail.GetSIAudit(this.StartDate, this.EndDate);
            Comment = "";
            RowModified = false; //indicates if the audited row has been modified or newly added

            //generating one audit report row for each changed field
            nCount = dtSIAudit.Rows.Count;
            for (int i = 0; i < nCount; i++)
            {
                if ((DateTime)dtSIAudit.Rows[i][ARTIFICIAL.MirrorDate] < this.StartDate)
                    continue;
                if (i == dtSIAudit.Rows.Count - 1)
                {
                    RowModified = false;
                }
                else
                {
                    //check for newly added TRs
                    if (dtSIAudit.Rows[i][TABLE.PatientSpecialInstruction.PatientSpecialInstructionGuid].ToString() != dtSIAudit.Rows[i + 1][TABLE.PatientSpecialInstruction.PatientSpecialInstructionGuid].ToString())
                    {
                        RowModified = false;
                    }
                    //checking for inactivated TRs
                    else if (dtSIAudit.Rows[i][TABLE.PatientSpecialInstruction.RecordStatusCode].ToString() == System.Convert.ToString(Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Inactive)) &&
                        dtSIAudit.Rows[i + 1][TABLE.PatientSpecialInstruction.RecordStatusCode].ToString() == System.Convert.ToString(Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active)))
                    {
                        RowModified = true;
                    }
                    else
                    {
                        continue;
                    }
                }

                DataRow drAuditReport = this._dtAuditReport.NewRow();

                drAuditReport[ARTIFICIAL.AuditType] = AuditTrailConst.AuditTrailType.TransfusionRequirement;
                drAuditReport[ARTIFICIAL.AuditIdentifier] = dtSIAudit.Rows[i][TABLE.Patient.PatientLastName].ToString().Trim() + ", " +
                    dtSIAudit.Rows[i][TABLE.Patient.PatientFirstName].ToString().Trim() + " (" +
                    dtSIAudit.Rows[i][TABLE.Patient.DisplayVistaPatientId].ToString().Trim() + ")";
                drAuditReport[ARTIFICIAL.ChangedFieldName] = "Status Indicator for Special Instruction (" + dtSIAudit.Rows[i][TABLE.PatientSpecialInstruction.PatientSpecialInstructions].ToString() + ")";

                if (RowModified == false)
                {
                    drAuditReport[ARTIFICIAL.OldValue] = "None";
                    drAuditReport[ARTIFICIAL.NewValue] = "Active";
                }
                else
                {
                    if (dtSIAudit.Rows[i][TABLE.PatientSpecialInstruction.RecordStatusCode].ToString().Trim() == Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active).ToString())
                    {
                        drAuditReport[ARTIFICIAL.OldValue] = "Inactive";
                        drAuditReport[ARTIFICIAL.NewValue] = "Active";
                    }
                    else
                    {
                        drAuditReport[ARTIFICIAL.OldValue] = "Active";
                        drAuditReport[ARTIFICIAL.NewValue] = "Inactive";
                    }
                }


                try
                {
                    BOL.VbecsUser LastUpdateUser = new BOL.VbecsUser(dtSIAudit.Rows[i][TABLE.PatientSpecialInstruction.LastUpdateUser].ToString().Trim());
                    if (LastUpdateUser.UserName == null)
                    {
                        drAuditReport[ARTIFICIAL.AuditUser] = dtSIAudit.Rows[i][TABLE.PatientSpecialInstruction.LastUpdateUser];
                    }
                    else
                    {
                        drAuditReport[ARTIFICIAL.AuditUser] = LastUpdateUser.UserName;
                    }
                }
                catch
                {
                    drAuditReport[ARTIFICIAL.AuditUser] = dtSIAudit.Rows[i][TABLE.PatientSpecialInstruction.LastUpdateUser];
                }

                drAuditReport[ARTIFICIAL.AuditDate] = dtSIAudit.Rows[i][ARTIFICIAL.MirrorDate];

                if (dtSIAudit.Rows[i].IsNull(TABLE.PatientSpecialInstruction.InactivationComment) == false)
                {
                    Comment += dtSIAudit.Rows[i][TABLE.PatientSpecialInstruction.InactivationComment].ToString().Trim();
                }

                drAuditReport[ARTIFICIAL.Comment] = Comment.Trim();
                Comment = "";
                this._dtAuditReport.Rows.Add(drAuditReport);
            }
        }

        /// <summary>
        /// Generates the DataTable used as a source for BloodUnitFinance Audit Report
        /// </summary>
        private void GetBloodUnitFinanceAuditReport()
        {
            bool RowModified = false;
            DataTable dtBloodUnitFinanceAudit = DAL.AuditTrail.GetBloodUnitFinanceAudit(this.StartDate, this.EndDate);
            int nCount = 0;

            //generating one audit report row for each changed field
            nCount = dtBloodUnitFinanceAudit.Rows.Count;
            for (int i = 0; i < nCount; i++)
            {
                if ((DateTime)dtBloodUnitFinanceAudit.Rows[i][ARTIFICIAL.MirrorDate] < this.StartDate)
                    continue;
                if (i == dtBloodUnitFinanceAudit.Rows.Count - 1)
                {
                    RowModified = false;
                }
                else
                {
                    //do not compare records of two different blood units
                    //if the record has just been added and there is no history of changes
                    if (dtBloodUnitFinanceAudit.Rows[i][TABLE.BloodUnitFinance.BloodUnitFinanceGuid].ToString() != dtBloodUnitFinanceAudit.Rows[i + 1][TABLE.BloodUnitFinance.BloodUnitFinanceGuid].ToString())
                    {
                        RowModified = false;
                    }
                    else
                    {
                        RowModified = true;
                    }
                }

                foreach (DataColumn dtCol in dtBloodUnitFinanceAudit.Columns)
                {
                    if (dtCol.ColumnName == TABLE.BloodUnitFinance.UnitCost ||
                        dtCol.ColumnName == TABLE.BloodUnitFinance.ReturnCreditAmount ||
                        dtCol.ColumnName == TABLE.BloodUnitFinance.SpecialTestingCost)
                    {
                        //skip columns that have not changed the value
                        if (RowModified == true)
                        {
                            if (dtBloodUnitFinanceAudit.Rows[i][dtCol].Equals(dtBloodUnitFinanceAudit.Rows[i + 1][dtCol]) ||
                                (dtBloodUnitFinanceAudit.Rows[i].IsNull(dtCol) == true &&
                                 dtBloodUnitFinanceAudit.Rows[i + 1].IsNull(dtCol) == true) ||
                                (int)dtBloodUnitFinanceAudit.Rows[i][TABLE.BloodUnitTest.LastUpdateFunctionId] != (int)Common.UpdateFunction.UC014FrmEditUnitFinancial)
                            {
                                continue;
                            }
                        }
                        else
                        {
                            //we do not have to audit new entries for these two columns
                            if (dtCol.ColumnName == TABLE.BloodUnitFinance.UnitCost ||
                                dtCol.ColumnName == TABLE.BloodUnitFinance.ReturnCreditAmount ||
                                dtBloodUnitFinanceAudit.Rows[i].IsNull(dtCol) == true ||
                                //CR 2714 - filtering out records that are created outside of UC_14
                                (int)dtBloodUnitFinanceAudit.Rows[i][TABLE.BloodUnitTest.LastUpdateFunctionId] != (int)Common.UpdateFunction.UC014FrmEditUnitFinancial)
                            {
                                continue;
                            }
                        }

                        DataRow drAuditReport = this._dtAuditReport.NewRow();
                        drAuditReport[ARTIFICIAL.AuditType] = AuditTrailConst.AuditTrailType.BloodUnitFinance;
                        drAuditReport[ARTIFICIAL.AuditIdentifier] = dtBloodUnitFinanceAudit.Rows[i][TABLE.BloodUnit.EyeReadableUnitId].ToString().Trim() + ", " + dtBloodUnitFinanceAudit.Rows[i][TABLE.BloodProduct.ProductCode].ToString().Trim() + ", " + dtBloodUnitFinanceAudit.Rows[i][TABLE.BloodProduct.ProductName].ToString().Trim();

                        if (dtCol.ColumnName == TABLE.BloodUnitFinance.UnitCost)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Unit Base Cost";
                        }
                        else if (dtCol.ColumnName == TABLE.BloodUnitFinance.ReturnCreditAmount)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Return Credit";
                        }
                        else if (dtCol.ColumnName == TABLE.BloodUnitFinance.SpecialTestingCost)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Special Test Cost";
                        }

                        if (RowModified == false)
                        {
                            drAuditReport[ARTIFICIAL.OldValue] = "None";
                        }
                        else
                        {
                            if (dtBloodUnitFinanceAudit.Rows[i + 1].IsNull(dtCol) == false)
                            {
                                drAuditReport[ARTIFICIAL.OldValue] = ((decimal)dtBloodUnitFinanceAudit.Rows[i + 1][dtCol]).ToString("C");
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.OldValue] = "None";
                            }
                        }

                        if (dtBloodUnitFinanceAudit.Rows[i].IsNull(dtCol) == false)
                        {
                            drAuditReport[ARTIFICIAL.NewValue] = ((decimal)dtBloodUnitFinanceAudit.Rows[i][dtCol]).ToString("C");
                        }
                        else
                        {
                            drAuditReport[ARTIFICIAL.NewValue] = "None";
                        }

                        try
                        {
                            BOL.VbecsUser LastUpdateUser = new BOL.VbecsUser(dtBloodUnitFinanceAudit.Rows[i][TABLE.SpecimenTestThreshold.LastUpdateUser].ToString().Trim());
                            if (LastUpdateUser.UserName == null)
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = dtBloodUnitFinanceAudit.Rows[i][TABLE.SpecimenTestThreshold.LastUpdateUser];
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = LastUpdateUser.UserName;
                            }
                        }
                        catch
                        {
                            drAuditReport[ARTIFICIAL.AuditUser] = dtBloodUnitFinanceAudit.Rows[i][TABLE.SpecimenTestThreshold.LastUpdateUser];
                        }

                        drAuditReport[ARTIFICIAL.AuditDate] = dtBloodUnitFinanceAudit.Rows[i][ARTIFICIAL.MirrorDate];
                        drAuditReport[ARTIFICIAL.Comment] = "";

                        this._dtAuditReport.Rows.Add(drAuditReport);
                    }
                }
            }
        }

        /// <summary>
        /// Generates the DataTable used as a source for Local Shipper Audit Report
        /// </summary>
        private void GetLocalSupplierAuditReport()
        {
            bool RowModified = false;
            DataTable dtLocalSupplierAudit = DAL.AuditTrail.GetLocalSupplierAudit(this.StartDate, this.EndDate);
            int nCount = 0;
            //generating one audit report row for each changed field
            nCount = dtLocalSupplierAudit.Rows.Count;
            for (int i = 0; i < nCount; i++)
            {
                if ((DateTime)dtLocalSupplierAudit.Rows[i][ARTIFICIAL.MirrorDate] < this.StartDate)
                    continue;
                if (i == dtLocalSupplierAudit.Rows.Count - 1)
                {
                    RowModified = false;
                }
                else
                {
                    //do not compare records of two different local Shippers
                    //if the record has just been added and there is no history of changes
                    if (dtLocalSupplierAudit.Rows[i][TABLE.LocalSupplier.LocalSupplierGuid].ToString() != dtLocalSupplierAudit.Rows[i + 1][TABLE.LocalSupplier.LocalSupplierGuid].ToString())
                    {
                        RowModified = false;
                    }
                    else
                    {
                        RowModified = true;
                    }
                }

                foreach (DataColumn dtCol in dtLocalSupplierAudit.Columns)
                {
                    if (dtCol.ColumnName.ToUpper() != ARTIFICIAL.Source &&
                        dtCol.ColumnName != ARTIFICIAL.MirrorId &&
                        dtCol.ColumnName != ARTIFICIAL.MirrorDate &&
                        dtCol.ColumnName != TABLE.LocalSupplier.LastUpdateDate &&
                        dtCol.ColumnName != TABLE.LocalSupplier.LastUpdateUser &&
                        dtCol.ColumnName != TABLE.LocalSupplier.LastUpdateFunctionId &&
                        dtCol.ColumnName != TABLE.LocalSupplier.LocalSupplierGuid &&
                        dtCol.ColumnName != TABLE.LocalSupplier.FacilityName &&
                        dtCol.ColumnName != TABLE.LocalSupplier.FdaRegistrationNumber)
                    {
                        //skip columns that have not changed the value
                        if (RowModified == true)
                        {
                            if (dtLocalSupplierAudit.Rows[i][dtCol].Equals(dtLocalSupplierAudit.Rows[i + 1][dtCol]) ||
                                (dtLocalSupplierAudit.Rows[i].IsNull(dtCol) && dtLocalSupplierAudit.Rows[i + 1].IsNull(dtCol)))
                            {
                                continue;
                            }
                        }
                        else
                        {
                            if (dtLocalSupplierAudit.Rows[i][dtCol].ToString().Trim() == "")
                                continue;
                        }

                        DataRow drAuditReport = this._dtAuditReport.NewRow();
                        drAuditReport[ARTIFICIAL.AuditType] = AuditTrailConst.AuditTrailType.LocalSupplier;
                        drAuditReport[ARTIFICIAL.AuditIdentifier] = dtLocalSupplierAudit.Rows[i][TABLE.LocalSupplier.FacilityName].ToString().Trim() + " (" + dtLocalSupplierAudit.Rows[i][TABLE.LocalSupplier.FdaRegistrationNumber].ToString().Trim() + ")";

                        if (dtCol.ColumnName == TABLE.LocalSupplier.LabelUnitIdTrimIndicator ||
                            dtCol.ColumnName == TABLE.LocalSupplier.TestingIndicator ||
                            dtCol.ColumnName == TABLE.LocalSupplier.CollectionIndicator)
                        {
                            if (dtLocalSupplierAudit.Rows[i].IsNull(dtCol) == false)
                            {
                                if ((bool)dtLocalSupplierAudit.Rows[i][dtCol] == false)
                                {
                                    drAuditReport[ARTIFICIAL.NewValue] = "No";
                                }
                                else
                                {
                                    drAuditReport[ARTIFICIAL.NewValue] = "Yes";
                                }
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "None";
                            }

                            if (RowModified == true)
                            {
                                if (dtLocalSupplierAudit.Rows[i + 1].IsNull(dtCol) == false)
                                {
                                    if ((bool)dtLocalSupplierAudit.Rows[i + 1][dtCol] == false)
                                    {
                                        drAuditReport[ARTIFICIAL.OldValue] = "No";
                                    }
                                    else
                                    {
                                        drAuditReport[ARTIFICIAL.OldValue] = "Yes";
                                    }
                                }
                                else
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "None";
                                }
                            }

                            if (dtCol.ColumnName == TABLE.LocalSupplier.LabelUnitIdTrimIndicator)
                            {
                                drAuditReport[ARTIFICIAL.ChangedFieldName] = "Alpha characters indicator";
                            }
                            else if (dtCol.ColumnName == TABLE.LocalSupplier.TestingIndicator)
                            {
                                drAuditReport[ARTIFICIAL.ChangedFieldName] = "Testing Facility Indicator";
                            }
                            else if (dtCol.ColumnName == TABLE.LocalSupplier.CollectionIndicator)
                            {
                                drAuditReport[ARTIFICIAL.ChangedFieldName] = "Collection Facility Indicator";
                            }
                        }
                        else if (dtCol.ColumnName == TABLE.LocalSupplier.RecordStatusCode)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Active Facility Indicator";
                            if (dtLocalSupplierAudit.Rows[i][dtCol].ToString() == System.Convert.ToString(Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.LocalEntry)))
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "Active";
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "Inactive";
                            }

                            if (RowModified == true)
                            {
                                if (dtLocalSupplierAudit.Rows[i + 1][dtCol].ToString() == Convert.ToString(Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.LocalEntry)))
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "Active";
                                }
                                else
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "Inactive";
                                }
                            }
                        }
                        else
                        {
                            if (dtCol.ColumnName == TABLE.Address.AddressLine1)
                            {
                                drAuditReport[ARTIFICIAL.ChangedFieldName] = "Address Line 1";
                            }
                            else if (dtCol.ColumnName == TABLE.Address.AddressLine2)
                            {
                                drAuditReport[ARTIFICIAL.ChangedFieldName] = "Address Line 2";
                            }
                            else if (dtCol.ColumnName == TABLE.Address.AddressLine3)
                            {
                                drAuditReport[ARTIFICIAL.ChangedFieldName] = "Address Line 3";
                            }
                            else if (dtCol.ColumnName == TABLE.Address.AddressCity)
                            {
                                drAuditReport[ARTIFICIAL.ChangedFieldName] = ARTIFICIAL.City;
                            }
                            else if (dtCol.ColumnName == TABLE.Address.AddressZipCode)
                            {
                                drAuditReport[ARTIFICIAL.ChangedFieldName] = "Zip Code";
                            }
                            else if (dtCol.ColumnName == TABLE.Address.StateCode)
                            {
                                drAuditReport[ARTIFICIAL.ChangedFieldName] = ARTIFICIAL.State;
                            }
                            else if (dtCol.ColumnName == TABLE.LocalSupplier.FaxNumber)
                            {
                                drAuditReport[ARTIFICIAL.ChangedFieldName] = ARTIFICIAL.Fax;
                            }
                            else if (dtCol.ColumnName == TABLE.LocalSupplier.PhoneNumber)
                            {
                                drAuditReport[ARTIFICIAL.ChangedFieldName] = ARTIFICIAL.Phone;
                            }
                            else if (dtCol.ColumnName == TABLE.LocalSupplier.FacilityPrefix)
                            {
                                drAuditReport[ARTIFICIAL.ChangedFieldName] = "Eye-readable prefix";
                            }
                            else if (dtCol.ColumnName == TABLE.LocalSupplier.IccbbaRegistrationNumber)
                            {
                                drAuditReport[ARTIFICIAL.ChangedFieldName] = "ICCBBA Registration Number";
                            }

                            if (dtLocalSupplierAudit.Rows[i][dtCol].ToString().Trim() != "")
                                drAuditReport[ARTIFICIAL.NewValue] = dtLocalSupplierAudit.Rows[i][dtCol].ToString().Trim();
                            else
                                drAuditReport[ARTIFICIAL.NewValue] = "None";

                            if (RowModified == true)
                            {
                                if (dtLocalSupplierAudit.Rows[i + 1][dtCol].ToString().Trim() != "")
                                    drAuditReport[ARTIFICIAL.OldValue] = dtLocalSupplierAudit.Rows[i + 1][dtCol].ToString().Trim();
                                else
                                    drAuditReport[ARTIFICIAL.OldValue] = "None";
                            }
                        }

                        if (RowModified == false)
                        {
                            drAuditReport[ARTIFICIAL.OldValue] = "None";
                        }

                        try
                        {
                            BOL.VbecsUser LastUpdateUser = new BOL.VbecsUser(dtLocalSupplierAudit.Rows[i][TABLE.LocalSupplier.LastUpdateUser].ToString().Trim());
                            if (LastUpdateUser.UserName == null)
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = dtLocalSupplierAudit.Rows[i][TABLE.LocalSupplier.LastUpdateUser];
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = LastUpdateUser.UserName;
                            }
                        }
                        catch
                        {
                            drAuditReport[ARTIFICIAL.AuditUser] = dtLocalSupplierAudit.Rows[i][TABLE.LocalSupplier.LastUpdateUser];
                        }

                        drAuditReport[ARTIFICIAL.AuditDate] = dtLocalSupplierAudit.Rows[i][ARTIFICIAL.MirrorDate];
                        drAuditReport[ARTIFICIAL.Comment] = "";

                        this._dtAuditReport.Rows.Add(drAuditReport);
                    }
                }
            }
        }

        /// <summary>
        /// Generates the DataTable used as a source for Blood Product Audit Report
        /// </summary>
        private void GetBloodProductAuditReport()
        {
            bool RowModified = false;
            //HCPCS code for blood product
            DataTable dtHcpcsAudit = DAL.AuditTrail.GetHcpcsAudit(this.StartDate, this.EndDate);
            int nCount = 0;
            //generating one audit report row for each changed field
            nCount = dtHcpcsAudit.Rows.Count;
            for (int i = 0; i < nCount; i++)
            {
                if ((DateTime)dtHcpcsAudit.Rows[i][ARTIFICIAL.MirrorDate] < this.StartDate)
                    continue;
                if (i == dtHcpcsAudit.Rows.Count - 1)
                {
                    RowModified = false;
                }
                else
                {
                    //new HCPCS code added
                    if (dtHcpcsAudit.Rows[i][TABLE.HcpcsAssignment.BloodProductGuid].ToString() != dtHcpcsAudit.Rows[i + 1][TABLE.HcpcsAssignment.BloodProductGuid].ToString() &&
                        dtHcpcsAudit.Rows[i][TABLE.HcpcsAssignment.HcpcsAssignmentGuid].ToString() != dtHcpcsAudit.Rows[i + 1][TABLE.HcpcsAssignment.HcpcsAssignmentGuid].ToString() &&
                        (int)dtHcpcsAudit.Rows[i][ARTIFICIAL.MirrorId] != 0)
                    {
                        RowModified = false;
                    }
                    //if the record has just been added and there is no history of changes
                    else if (dtHcpcsAudit.Rows[i][TABLE.HcpcsAssignment.BloodProductGuid].ToString() != dtHcpcsAudit.Rows[i + 1][TABLE.HcpcsAssignment.BloodProductGuid].ToString() &&
                             dtHcpcsAudit.Rows[i][TABLE.HcpcsAssignment.HcpcsAssignmentGuid].ToString() != dtHcpcsAudit.Rows[i + 1][TABLE.HcpcsAssignment.HcpcsAssignmentGuid].ToString() &&
                        (int)dtHcpcsAudit.Rows[i][ARTIFICIAL.MirrorId] == 0)
                    {
                        RowModified = false;
                    }
                    else
                    {
                        RowModified = true;
                    }
                }

                DataRow drAuditReport = this._dtAuditReport.NewRow();

                drAuditReport[ARTIFICIAL.AuditType] = AuditTrailConst.AuditTrailType.BloodProduct;
                drAuditReport[ARTIFICIAL.AuditIdentifier] = dtHcpcsAudit.Rows[i][TABLE.BloodProduct.ProductName].ToString().Trim() + " (" + dtHcpcsAudit.Rows[i][TABLE.BloodProduct.ProductCode].ToString().Trim() + ")";
                drAuditReport[ARTIFICIAL.ChangedFieldName] = "HCPCS Code: " + dtHcpcsAudit.Rows[i][TABLE.HcpcsAssignment.HcpcsCode].ToString().Trim();

                if (RowModified == false)
                {
                    if (dtHcpcsAudit.Rows[i].IsNull(TABLE.HcpcsAssignment.UnAssignedDate) == true)
                    {
                        drAuditReport[ARTIFICIAL.OldValue] = "None";
                        drAuditReport[ARTIFICIAL.NewValue] = "Assigned";
                    }
                    else
                    {
                        drAuditReport[ARTIFICIAL.OldValue] = "Assigned";
                        drAuditReport[ARTIFICIAL.NewValue] = "Unassigned";
                    }
                }
                else
                {
                    if (dtHcpcsAudit.Rows[i].IsNull(TABLE.HcpcsAssignment.UnAssignedDate) == false)
                    {
                        drAuditReport[ARTIFICIAL.OldValue] = "Assigned";
                        drAuditReport[ARTIFICIAL.NewValue] = "Unassigned";
                    }
                    else
                    {
                        drAuditReport[ARTIFICIAL.OldValue] = "Unassigned";
                        drAuditReport[ARTIFICIAL.NewValue] = "Assigned";
                    }
                }

                try
                {
                    BOL.VbecsUser LastUpdateUser = new BOL.VbecsUser(dtHcpcsAudit.Rows[i][TABLE.SpecimenTestThreshold.LastUpdateUser].ToString().Trim());
                    if (LastUpdateUser.UserName == null)
                    {
                        drAuditReport[ARTIFICIAL.AuditUser] = dtHcpcsAudit.Rows[i][TABLE.SpecimenTestThreshold.LastUpdateUser];
                    }
                    else
                    {
                        drAuditReport[ARTIFICIAL.AuditUser] = LastUpdateUser.UserName;
                    }
                }
                catch
                {
                    drAuditReport[ARTIFICIAL.AuditUser] = dtHcpcsAudit.Rows[i][TABLE.SpecimenTestThreshold.LastUpdateUser];
                }

                drAuditReport[ARTIFICIAL.AuditDate] = dtHcpcsAudit.Rows[i][ARTIFICIAL.MirrorDate];
                drAuditReport[ARTIFICIAL.Comment] = "";
                this._dtAuditReport.Rows.Add(drAuditReport);
            }

            //getting Product Shipper audit and appending it to Local Shipper audit
            DataTable dtProductShipperAudit = DAL.AuditTrail.GetProductShipperAudit(this.StartDate, this.EndDate);

            //generating one audit report row for each changed field
            nCount = dtProductShipperAudit.Rows.Count;
            for (int i = 0; i < nCount; i++)
            {
                if ((DateTime)dtProductShipperAudit.Rows[i][ARTIFICIAL.MirrorDate] < this.StartDate)
                    continue;
                if (i == dtProductShipperAudit.Rows.Count - 1)
                {
                    RowModified = false;
                }
                else
                {
                    //do not compare records of two different product Shippers
                    if (dtProductShipperAudit.Rows[i][TABLE.ProductShipper.ProductShipperGuid].ToString() != dtProductShipperAudit.Rows[i + 1][TABLE.ProductShipper.ProductShipperGuid].ToString() &&
                             dtProductShipperAudit.Rows[i][TABLE.ProductShipper.BloodProductGuid].ToString() != dtProductShipperAudit.Rows[i + 1][TABLE.ProductShipper.BloodProductGuid].ToString())
                    {
                        RowModified = false;
                    }
                    else
                    {
                        RowModified = true;
                    }
                }

                foreach (DataColumn dtCol in dtProductShipperAudit.Columns)
                {
                    if (dtCol.ColumnName == TABLE.ProductShipper.ReturnCreditPercentage ||
                        dtCol.ColumnName == TABLE.ProductShipper.RecordStatusCode ||
                        dtCol.ColumnName == TABLE.ProductShipper.ProductCost)
                    {
                        //skip columns that have not changed the value
                        if (RowModified == true)
                        {
                            if (dtProductShipperAudit.Rows[i][dtCol].Equals(dtProductShipperAudit.Rows[i + 1][dtCol]))
                            {
                                continue;
                            }
                        }

                        DataRow drAuditReport = this._dtAuditReport.NewRow();
                        drAuditReport[ARTIFICIAL.AuditType] = AuditTrailConst.AuditTrailType.BloodProduct;
                        drAuditReport[ARTIFICIAL.AuditIdentifier] = dtProductShipperAudit.Rows[i][TABLE.BloodProduct.ProductName].ToString().Trim() + " (" + dtProductShipperAudit.Rows[i][TABLE.BloodProduct.ProductCode].ToString().Trim() + ")";

                        if (dtCol.ColumnName == TABLE.ProductShipper.ReturnCreditPercentage)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = dtProductShipperAudit.Rows[i][TABLE.LocalSupplier.FacilityName].ToString().Trim() + ", " + dtProductShipperAudit.Rows[i][TABLE.LocalSupplier.FdaRegistrationNumber].ToString().Trim() + " (Return Credit Percentage)";
                            drAuditReport[ARTIFICIAL.NewValue] = dtProductShipperAudit.Rows[i][TABLE.ProductShipper.ReturnCreditPercentage];
                            if (RowModified == false)
                            {
                                drAuditReport[ARTIFICIAL.OldValue] = "None";
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.OldValue] = dtProductShipperAudit.Rows[i + 1][TABLE.ProductShipper.ReturnCreditPercentage];
                            }
                        }
                        else if (dtCol.ColumnName == TABLE.ProductShipper.ProductCost)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = dtProductShipperAudit.Rows[i][TABLE.LocalSupplier.FacilityName].ToString().Trim() + ", " + dtProductShipperAudit.Rows[i][TABLE.LocalSupplier.FdaRegistrationNumber].ToString().Trim() + " (Cost)";
                            drAuditReport[ARTIFICIAL.NewValue] = dtProductShipperAudit.Rows[i][TABLE.ProductShipper.ProductCost];
                            if (RowModified == false)
                            {
                                drAuditReport[ARTIFICIAL.OldValue] = "None";
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.OldValue] = dtProductShipperAudit.Rows[i + 1][TABLE.ProductShipper.ProductCost];
                            }
                        }
                        else if (dtCol.ColumnName == TABLE.ProductShipper.RecordStatusCode)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = dtProductShipperAudit.Rows[i][TABLE.LocalSupplier.FacilityName].ToString().Trim() + ", " + dtProductShipperAudit.Rows[i][TABLE.LocalSupplier.FdaRegistrationNumber].ToString().Trim() + " (Status Indicator)";
                            if (Common.Utility.GetRecordStatusCodeFromString(dtProductShipperAudit.Rows[i][TABLE.ProductShipper.RecordStatusCode].ToString().Trim()) == Common.RecordStatusCode.Active)
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "Active";
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "Inactive";
                            }

                            if (RowModified == false)
                            {
                                drAuditReport[ARTIFICIAL.OldValue] = "None";
                            }
                            else
                            {
                                if (Common.Utility.GetRecordStatusCodeFromString(dtProductShipperAudit.Rows[i + 1][TABLE.ProductShipper.RecordStatusCode].ToString().Trim()) == Common.RecordStatusCode.Active)
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "Active";
                                }
                                else
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "Inactive";
                                }
                            }
                        }

                        try
                        {
                            BOL.VbecsUser LastUpdateUser = new BOL.VbecsUser(dtProductShipperAudit.Rows[i][TABLE.ProductShipper.LastUpdateUser].ToString().Trim());
                            if (LastUpdateUser.UserName == null)
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = dtProductShipperAudit.Rows[i][TABLE.ProductShipper.LastUpdateUser];
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = LastUpdateUser.UserName;
                            }
                        }
                        catch
                        {
                            drAuditReport[ARTIFICIAL.AuditUser] = dtProductShipperAudit.Rows[i][TABLE.ProductShipper.LastUpdateUser];
                        }

                        drAuditReport[ARTIFICIAL.AuditDate] = dtProductShipperAudit.Rows[i][ARTIFICIAL.MirrorDate];
                        drAuditReport[ARTIFICIAL.Comment] = "";

                        this._dtAuditReport.Rows.Add(drAuditReport);
                    }
                }
            }
        }

        /// <summary>
        /// Generates the DataTable used as a source for Division Audit Report (including Invoice Text changes)
        /// </summary>
        private void GetDivisionAuditReport()
        {
            bool RowModified = false;
            DataTable dtDivisionAudit = DAL.AuditTrail.GetDivisionAudit(this.StartDate, this.EndDate);
            int nCount = 0;
            //generating one audit report row for each changed field
            nCount = dtDivisionAudit.Rows.Count;
            for (int i = 0; i < nCount; i++)
            {
                if ((DateTime)dtDivisionAudit.Rows[i][ARTIFICIAL.MirrorDate] < this.StartDate)
                    continue;
                if (i == dtDivisionAudit.Rows.Count - 1)
                {
                    RowModified = false;
                }
                else
                {
                    //do not compare records of two different blood units
                    //if the record has just been added and there is no history of changes
                    if (dtDivisionAudit.Rows[i][TABLE.VamcDivision.DivisionCode].ToString() != dtDivisionAudit.Rows[i + 1][TABLE.VamcDivision.DivisionCode].ToString())
                    {
                        RowModified = false;
                    }
                    else
                    {
                        RowModified = true;
                    }
                }

                foreach (DataColumn dtCol in dtDivisionAudit.Columns)
                {
                    if (dtCol.ColumnName == TABLE.VamcDivision.ServiceTypeIndicator ||
                        dtCol.ColumnName == TABLE.Facility.FacilityName ||
                        dtCol.ColumnName == TABLE.VamcDivision.RecordStatusCode ||
                        dtCol.ColumnName == TABLE.VamcDivision.LabelPrinterUseIndicator ||
                        dtCol.ColumnName == TABLE.VamcDivision.LabelPrinterTcpPortNumber ||
                        dtCol.ColumnName == TABLE.VamcDivision.LabelPrinterIPAddress ||
                        dtCol.ColumnName == TABLE.VamcDivision.LabelPrinterComPortNumber ||
                        dtCol.ColumnName == TABLE.VamcDivision.AccessionAreaId ||
                        dtCol.ColumnName == TABLE.VamcDivision.LockInactivityTimeoutMin ||
                        dtCol.ColumnName == TABLE.TimeZone.TimeZoneName ||
                        dtCol.ColumnName == TABLE.DaylightSavingsType.DaylightSavingsTypeText ||
                        dtCol.ColumnName == TABLE.VamcDivision.ElectronicCrossmatchIndicator ||
                        dtCol.ColumnName == TABLE.VamcDivision.IccbbaRegistrationNumber ||
                        dtCol.ColumnName == TABLE.VamcDivision.MaximumSpecimenExpirationDays ||
                        dtCol.ColumnName == TABLE.VamcDivision.MedicalDirectorName ||
                        dtCol.ColumnName == TABLE.VamcDivision.LoginMessageText ||
                        dtCol.ColumnName == TABLE.VamcDivision.InvoiceTemplateText ||
                        dtCol.ColumnName == TABLE.AntibodyScreenDisplay.AntibodyScreenDisplayText ||
                        dtCol.ColumnName == TABLE.VamcDivision.DaylightSavingsStartDate ||
                        dtCol.ColumnName == TABLE.VamcDivision.DaylightSavingsEndDate ||
                        dtCol.ColumnName == TABLE.VamcDivision.PrinterName ||
                        dtCol.ColumnName == TABLE.VamcDivision.ReactivityPhaseCode)
                    {
                        //skip columns that have not changed the value
                        if (RowModified == true)
                        {
                            if (dtDivisionAudit.Rows[i][dtCol].Equals(dtDivisionAudit.Rows[i + 1][dtCol]) ||
                                (dtDivisionAudit.Rows[i].IsNull(dtCol) && dtDivisionAudit.Rows[i + 1].IsNull(dtCol)) ||
                                (dtDivisionAudit.Rows[i].IsNull(dtCol) && dtDivisionAudit.Rows[i + 1][dtCol].ToString().Trim() == "") ||
                                (dtDivisionAudit.Rows[i][dtCol].ToString().Trim() == "" && dtDivisionAudit.Rows[i + 1][dtCol].ToString().Trim() == ""))
                            {
                                continue;
                            }
                        }
                        else
                        {
                            if (dtDivisionAudit.Rows[i].IsNull(dtCol))
                                continue;
                        }

                        DataRow drAuditReport = this._dtAuditReport.NewRow();
                        drAuditReport[ARTIFICIAL.AuditType] = AuditTrailConst.AuditTrailType.MaintainDivision;
                        drAuditReport[ARTIFICIAL.AuditIdentifier] = dtDivisionAudit.Rows[i][TABLE.VamcDivision.DivisionName].ToString().Trim() + " (" + dtDivisionAudit.Rows[i][TABLE.VamcDivision.DivisionCode].ToString().Trim() + ")";
                        if (dtCol.ColumnName == TABLE.VamcDivision.ServiceTypeIndicator)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Full Service, Transfusion Only Indicator";
                        }
                        else if (dtCol.ColumnName == TABLE.Facility.FacilityName)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Facility Name";
                        }
                        else if (dtCol.ColumnName == TABLE.VamcDivision.RecordStatusCode)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Active Status";
                        }
                        else if (dtCol.ColumnName == TABLE.VamcDivision.LabelPrinterUseIndicator)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Full-Face-Label Printer Status";
                        }
                        else if (dtCol.ColumnName == TABLE.VamcDivision.LabelPrinterTcpPortNumber)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Full-Face-Label Printer Port Number";
                        }
                        else if (dtCol.ColumnName == TABLE.VamcDivision.LabelPrinterIPAddress)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Full-Face-Label Printer IP Address";
                        }
                        else if (dtCol.ColumnName == TABLE.VamcDivision.LabelPrinterComPortNumber)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Full Face Label Printer COM";
                        }
                        else if (dtCol.ColumnName == TABLE.VamcDivision.AccessionAreaId)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Accession Area Name";
                        }
                        else if (dtCol.ColumnName == TABLE.VamcDivision.LockInactivityTimeoutMin)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Lock Inactivity Timeout Setting";
                        }
                        else if (dtCol.ColumnName == TABLE.TimeZone.TimeZoneName)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Time Zone Setting";
                        }
                        else if (dtCol.ColumnName == TABLE.DaylightSavingsType.DaylightSavingsTypeText)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Daylight savings time Setting";
                        }
                        else if (dtCol.ColumnName == TABLE.VamcDivision.ElectronicCrossmatchIndicator)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Electronic Crossmatch Enabled Indicator";
                        }
                        else if (dtCol.ColumnName == TABLE.VamcDivision.IccbbaRegistrationNumber)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "ICCBBA Registration Number";
                        }
                        else if (dtCol.ColumnName == TABLE.VamcDivision.MaximumSpecimenExpirationDays)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Maximum Specimen Expiration Days";
                        }
                        else if (dtCol.ColumnName == TABLE.VamcDivision.MedicalDirectorName)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Blood Bank MD";
                        }
                        else if (dtCol.ColumnName == TABLE.VamcDivision.LoginMessageText)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Login Message";
                        }
                        else if (dtCol.ColumnName == TABLE.VamcDivision.InvoiceTemplateText)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Invoice Text";
                        }
                        else if (dtCol.ColumnName == TABLE.AntibodyScreenDisplay.AntibodyScreenDisplayText)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Antibody Screen Indicator";
                        }
                        else if (dtCol.ColumnName == TABLE.VamcDivision.DaylightSavingsStartDate)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Daylight savings time start date";
                        }
                        else if (dtCol.ColumnName == TABLE.VamcDivision.DaylightSavingsEndDate)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Daylight savings time end date";
                        }
                        else if (dtCol.ColumnName == TABLE.VamcDivision.PrinterName)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Report Printer Name";
                        }
                        //CR 2717
                        else if (dtCol.ColumnName == TABLE.VamcDivision.ReactivityPhaseCode)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "ABS and XM Testing Phases";
                        }

                        //previous value
                        if (RowModified == false)
                        {
                            drAuditReport[ARTIFICIAL.OldValue] = "None";
                        }
                        else
                        {
                            if (dtCol.ColumnName == TABLE.VamcDivision.ServiceTypeIndicator)
                            {
                                if ((bool)dtDivisionAudit.Rows[i + 1][dtCol] == true)
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "Full Service";
                                }
                                else
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "Transfusion Only";
                                }
                            }
                            else if (dtCol.ColumnName == TABLE.VamcDivision.ElectronicCrossmatchIndicator)
                            {
                                if ((bool)dtDivisionAudit.Rows[i + 1][dtCol] == true)
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "Enabled";
                                }
                                else
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "Disabled";
                                }
                            }
                            else if (dtCol.ColumnName == TABLE.VamcDivision.RecordStatusCode)
                            {
                                if (dtDivisionAudit.Rows[i + 1][dtCol].ToString() == Convert.ToString(Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Inactive)))
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "Inactive";
                                }
                                else
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "Active";
                                }
                            }
                            else if (dtCol.ColumnName == TABLE.VamcDivision.LabelPrinterUseIndicator)
                            {
                                if ((bool)dtDivisionAudit.Rows[i + 1][dtCol] == true)
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "Enabled";
                                }
                                else
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "Disabled";
                                }
                            }
                            else if (dtCol.ColumnName == TABLE.VamcDivision.AccessionAreaId)
                            {
                                //CR 3305
                                if (DAL.VAL.VistALink.EnsureAvailability())
                                {
                                    BOL.BusinessObjectSortedHashList accessionAreas =
                                        BOL.BloodBankAccessionArea.GetAllSortedByName();
                                    foreach (BOL.BloodBankAccessionArea bbArea in accessionAreas)
                                    {
                                        if (bbArea.ID == (int) dtDivisionAudit.Rows[i + 1][dtCol])
                                        {
                                            drAuditReport[ARTIFICIAL.OldValue] = bbArea.Name;
                                            break;
                                        }
                                    }
                                }
                                else 
                                    continue;
                            }
                            else if (dtCol.ColumnName == TABLE.VamcDivision.DaylightSavingsStartDate ||
                                dtCol.ColumnName == TABLE.VamcDivision.DaylightSavingsEndDate)
                            {
                                if (dtDivisionAudit.Rows[i + 1].IsNull(dtCol))
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "None";
                                }
                                else
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = ((DateTime)dtDivisionAudit.Rows[i + 1][dtCol]).ToString("d");
                                }
                            }
                            //CR 2717
                            else if (dtCol.ColumnName == TABLE.VamcDivision.ReactivityPhaseCode)
                            {
                                if (Common.Utility.GetReactivityPhaseCodeFromChar(Convert.ToChar(dtDivisionAudit.Rows[i + 1][dtCol])) == Common.ReactivityPhase.All)
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "All Phases";
                                }
                                else
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "AHG Only";
                                }
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.OldValue] = dtDivisionAudit.Rows[i + 1][dtCol];
                            }
                        }
                        if (drAuditReport[ARTIFICIAL.OldValue].ToString().Trim() == "")
                            drAuditReport[ARTIFICIAL.OldValue] = "None";

                        //new value
                        if (dtCol.ColumnName == TABLE.VamcDivision.ServiceTypeIndicator)
                        {
                            if ((bool)dtDivisionAudit.Rows[i][dtCol] == true)
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "Full Service";
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "Transfusion Only";
                            }
                        }
                        else if (dtCol.ColumnName == TABLE.VamcDivision.ElectronicCrossmatchIndicator)
                        {
                            if ((bool)dtDivisionAudit.Rows[i][dtCol] == true)
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "Enabled";
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "Disabled";
                            }
                        }
                        else if (dtCol.ColumnName == TABLE.VamcDivision.RecordStatusCode)
                        {
                            if (dtDivisionAudit.Rows[i][dtCol].ToString() == Convert.ToString(Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Inactive)))
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "Inactive";
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "Active";
                            }
                        }
                        else if (dtCol.ColumnName == TABLE.VamcDivision.LabelPrinterUseIndicator)
                        {
                            if ((bool)dtDivisionAudit.Rows[i][dtCol] == true)
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "Enabled";
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "Disabled";
                            }
                        }
                        else if (dtCol.ColumnName == TABLE.VamcDivision.AccessionAreaId)
                        {
                            //CR 3305
                            if (DAL.VAL.VistALink.EnsureAvailability())
                            {
                                BOL.BusinessObjectSortedHashList accessionAreas =
                                    BOL.BloodBankAccessionArea.GetAllSortedByName();
                                foreach (BOL.BloodBankAccessionArea bbArea in accessionAreas)
                                {
                                    if (bbArea.ID == (int) dtDivisionAudit.Rows[i][dtCol])
                                    {
                                        drAuditReport[ARTIFICIAL.NewValue] = bbArea.Name;
                                        break;
                                    }
                                }
                            }
                            else
                                continue;
                        }
                        else if (dtCol.ColumnName == TABLE.VamcDivision.DaylightSavingsStartDate ||
                                 dtCol.ColumnName == TABLE.VamcDivision.DaylightSavingsEndDate)
                        {
                            if (dtDivisionAudit.Rows[i].IsNull(dtCol))
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "None";
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = ((DateTime)dtDivisionAudit.Rows[i][dtCol]).ToString("d");
                            }
                        }
                        //CR 2717
                        else if (dtCol.ColumnName == TABLE.VamcDivision.ReactivityPhaseCode)
                        {
                            if (Common.Utility.GetReactivityPhaseCodeFromChar(Convert.ToChar(dtDivisionAudit.Rows[i][dtCol])) == Common.ReactivityPhase.All)
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "All Phases";
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "AHG Only";
                            }
                        }
                        else
                        {
                            drAuditReport[ARTIFICIAL.NewValue] = dtDivisionAudit.Rows[i][dtCol];
                        }

                        if (drAuditReport[ARTIFICIAL.NewValue].ToString().Trim() == "")
                            drAuditReport[ARTIFICIAL.NewValue] = "None";

                        try
                        {
                            BOL.VbecsUser LastUpdateUser = new BOL.VbecsUser(dtDivisionAudit.Rows[i][TABLE.SpecimenTestThreshold.LastUpdateUser].ToString().Trim());
                            if (LastUpdateUser.UserName == null)
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = dtDivisionAudit.Rows[i][TABLE.SpecimenTestThreshold.LastUpdateUser];
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = LastUpdateUser.UserName;
                            }
                        }
                        catch
                        {
                            drAuditReport[ARTIFICIAL.AuditUser] = dtDivisionAudit.Rows[i][TABLE.SpecimenTestThreshold.LastUpdateUser];
                        }
                        drAuditReport[ARTIFICIAL.AuditDate] = dtDivisionAudit.Rows[i][ARTIFICIAL.MirrorDate];
                        drAuditReport[ARTIFICIAL.Comment] = "";

                        this._dtAuditReport.Rows.Add(drAuditReport);
                    }
                }
            }
        }

        /// <summary>
        /// Generates the DataTable used as a source for Associated Institution Audit Report
        /// </summary>
        private void GetAssociatedInstitutionAuditReport()
        {
            bool RowModified = false;
            DataTable dtAssociatedInstitutionAudit = DAL.AuditTrail.GetAssociatedInstitutionAudit(this.StartDate, this.EndDate);
            int nCount = 0;
            //generating one audit report row for each changed field
            nCount = dtAssociatedInstitutionAudit.Rows.Count;
            for (int i = 0; i < nCount; i++)
            {
                if ((DateTime)dtAssociatedInstitutionAudit.Rows[i][ARTIFICIAL.MirrorDate] < this.StartDate)
                    continue;
                if (i == dtAssociatedInstitutionAudit.Rows.Count - 1)
                {
                    RowModified = false;
                }
                else
                {
                    //do not compare records of two different blood units
                    //if the record has just been added and there is no history of changes
                    if (dtAssociatedInstitutionAudit.Rows[i][TABLE.AssociatedInstitution.AssociatedInstitutionGuid].ToString() != dtAssociatedInstitutionAudit.Rows[i + 1][TABLE.AssociatedInstitution.AssociatedInstitutionGuid].ToString())
                    {
                        RowModified = false;
                    }
                    else
                    {
                        RowModified = true;
                    }
                }

                foreach (DataColumn dtCol in dtAssociatedInstitutionAudit.Columns)
                {
                    if (dtCol.ColumnName == TABLE.VamcDivision.RecordStatusCode)
                    {
                        //skip columns that have not changed the value
                        if (RowModified == true)
                        {
                            if (dtAssociatedInstitutionAudit.Rows[i][dtCol].Equals(dtAssociatedInstitutionAudit.Rows[i + 1][dtCol]) ||
                                (dtAssociatedInstitutionAudit.Rows[i].IsNull(dtCol) && dtAssociatedInstitutionAudit.Rows[i + 1].IsNull(dtCol)) ||
                                (dtAssociatedInstitutionAudit.Rows[i].IsNull(dtCol) && dtAssociatedInstitutionAudit.Rows[i + 1][dtCol].ToString().Trim() == "") ||
                                (dtAssociatedInstitutionAudit.Rows[i][dtCol].ToString().Trim() == "" && dtAssociatedInstitutionAudit.Rows[i + 1][dtCol].ToString().Trim() == ""))
                            {
                                continue;
                            }
                        }
                        else
                        {
                            if (dtAssociatedInstitutionAudit.Rows[i].IsNull(dtCol))
                                continue;
                        }

                        DataRow drAuditReport = this._dtAuditReport.NewRow();
                        drAuditReport[ARTIFICIAL.AuditType] = AuditTrailConst.AuditTrailType.MaintainDivision;
                        drAuditReport[ARTIFICIAL.AuditIdentifier] = dtAssociatedInstitutionAudit.Rows[i][TABLE.VamcDivision.DivisionName].ToString().Trim() + " (" + dtAssociatedInstitutionAudit.Rows[i][TABLE.VamcDivision.DivisionCode].ToString().Trim() + ")";
                        if (dtCol.ColumnName == TABLE.VamcDivision.RecordStatusCode)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Associated Institution Status (" + dtAssociatedInstitutionAudit.Rows[i][TABLE.AssociatedInstitution.InstitutionDivisionCode].ToString().Trim() + ")";
                        }

                        //previous value
                        if (RowModified == false)
                        {
                            drAuditReport[ARTIFICIAL.OldValue] = "None";
                        }
                        else
                        {
                            if (dtCol.ColumnName == TABLE.VamcDivision.RecordStatusCode)
                            {
                                if (dtAssociatedInstitutionAudit.Rows[i + 1][dtCol].ToString() == Convert.ToString(Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Inactive)))
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "Inactive";
                                }
                                else
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "Active";
                                }
                            }
                        }
                        if (drAuditReport[ARTIFICIAL.OldValue].ToString().Trim() == "")
                            drAuditReport[ARTIFICIAL.OldValue] = "None";

                        //new value
                        if (dtCol.ColumnName == TABLE.VamcDivision.RecordStatusCode)
                        {
                            if (dtAssociatedInstitutionAudit.Rows[i][dtCol].ToString() == Convert.ToString(Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Inactive)))
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "Inactive";
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "Active";
                            }
                        }

                        if (drAuditReport[ARTIFICIAL.NewValue].ToString().Trim() == "")
                            drAuditReport[ARTIFICIAL.NewValue] = "None";

                        try
                        {
                            BOL.VbecsUser LastUpdateUser = new BOL.VbecsUser(dtAssociatedInstitutionAudit.Rows[i][TABLE.SpecimenTestThreshold.LastUpdateUser].ToString().Trim());
                            if (LastUpdateUser.UserName == null)
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = dtAssociatedInstitutionAudit.Rows[i][TABLE.AssociatedInstitution.LastUpdateUser];
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = LastUpdateUser.UserName;
                            }
                        }
                        catch
                        {
                            drAuditReport[ARTIFICIAL.AuditUser] = dtAssociatedInstitutionAudit.Rows[i][TABLE.AssociatedInstitution.LastUpdateUser];
                        }
                        drAuditReport[ARTIFICIAL.AuditDate] = dtAssociatedInstitutionAudit.Rows[i][ARTIFICIAL.MirrorDate];
                        drAuditReport[ARTIFICIAL.Comment] = "";

                        this._dtAuditReport.Rows.Add(drAuditReport);
                    }
                }
            }
        }

        /// <summary>
        /// Generates the DataTable used as a source for Transfusion Requirement Audit Report
        /// </summary>
        private void GetProductModificationParameterAuditReport()
        {
            DataTable dtProductModificationParameterAudit = DAL.AuditTrail.GetProductModificationParameterAudit(this.StartDate, this.EndDate);
            bool RowModified = false; //indicates if the audited row has been modified or newly added
            int nCount = 0;
            //generating one audit report row for each changed field
            nCount = dtProductModificationParameterAudit.Rows.Count;
            for (int i = 0; i < nCount; i++)
            {
                if ((DateTime)dtProductModificationParameterAudit.Rows[i][ARTIFICIAL.MirrorDate] < this.StartDate)
                    continue;
                if (i == dtProductModificationParameterAudit.Rows.Count - 1)
                {
                    RowModified = false;
                }
                else
                {
                    //do not compare records of two different product modification parameters
                    //check for newly added product modification parameters
                    if (dtProductModificationParameterAudit.Rows[i][TABLE.ProductModificationParameter.ProductModificationParameterGuid].ToString() != dtProductModificationParameterAudit.Rows[i + 1][TABLE.ProductModificationParameter.ProductModificationParameterGuid].ToString())
                    {
                        RowModified = false;
                    }
                    else
                    {
                        RowModified = true;
                    }
                }

                foreach (DataColumn dtCol in dtProductModificationParameterAudit.Columns)
                {
                    if (dtCol.ColumnName == TABLE.ProductModificationParameter.RecordStatusCode ||
                        dtCol.ColumnName == TABLE.ProductModificationParameter.ProductModificationCost)
                    {
                        //skip columns that have not changed the value)
                        if (RowModified == true)
                        {
                            if (dtProductModificationParameterAudit.Rows[i][dtCol].Equals(dtProductModificationParameterAudit.Rows[i + 1][dtCol]))
                            {
                                continue;
                            }
                        }

                        DataRow drAuditReport = this._dtAuditReport.NewRow();

                        drAuditReport[ARTIFICIAL.AuditType] = AuditTrailConst.AuditTrailType.ProductModifications;
                        drAuditReport[ARTIFICIAL.AuditIdentifier] = dtProductModificationParameterAudit.Rows[i][TABLE.ProductModification.ProductModificationText].ToString().Trim();

                        if (dtCol.ColumnName == TABLE.ProductModificationParameter.RecordStatusCode)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Enabled/Disabled Indicator";
                        }
                        else
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = ARTIFICIAL.Cost;
                        }

                        //previous value
                        if (RowModified == false)
                        {
                            drAuditReport[ARTIFICIAL.OldValue] = "None";
                        }
                        else
                        {
                            if (dtCol.ColumnName == TABLE.ProductModificationParameter.RecordStatusCode)
                            {
                                if (dtProductModificationParameterAudit.Rows[i + 1][dtCol].ToString() == Convert.ToString(Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Inactive)))
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "Disabled";
                                }
                                else
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "Enabled";
                                }
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.OldValue] = Convert.ToDecimal(dtProductModificationParameterAudit.Rows[i + 1][dtCol]).ToString("C");
                            }
                        }

                        //new value
                        if (dtCol.ColumnName == TABLE.ProductModificationParameter.RecordStatusCode)
                        {
                            if (dtProductModificationParameterAudit.Rows[i][dtCol].ToString().Trim() == Convert.ToString(Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Inactive)))
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "Disabled";
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "Enabled";
                            }
                        }
                        else
                        {
                            drAuditReport[ARTIFICIAL.NewValue] = Convert.ToDecimal(dtProductModificationParameterAudit.Rows[i][dtCol]).ToString("C");
                        }

                        try
                        {
                            BOL.VbecsUser LastUpdateUser = new BOL.VbecsUser(dtProductModificationParameterAudit.Rows[i][TABLE.SpecimenTestThreshold.LastUpdateUser].ToString().Trim());
                            if (LastUpdateUser.UserName == null)
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = dtProductModificationParameterAudit.Rows[i][TABLE.SpecimenTestThreshold.LastUpdateUser];
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = LastUpdateUser.UserName;
                            }
                        }
                        catch
                        {
                            drAuditReport[ARTIFICIAL.AuditUser] = dtProductModificationParameterAudit.Rows[i][TABLE.SpecimenTestThreshold.LastUpdateUser];
                        }

                        drAuditReport[ARTIFICIAL.AuditDate] = dtProductModificationParameterAudit.Rows[i][ARTIFICIAL.MirrorDate];
                        drAuditReport[ARTIFICIAL.Comment] = "";
                        this._dtAuditReport.Rows.Add(drAuditReport);
                    }
                }
            }
        }

        /// <summary>
        /// Generates the DataTable used as a source for Antibody Audit Report
        /// </summary>
        private void GetAntibodyAuditReport()
        {
            bool RowModified = false;
            DataTable dtAntibodyAudit = DAL.AuditTrail.GetAntibodyAudit(this.StartDate, this.EndDate);
            dtAntibodyAudit.CaseSensitive = true;
            int nCount = 0;
            //generating one audit report row for each changed field
            nCount = dtAntibodyAudit.Rows.Count;
            for (int i = 0; i < nCount; i++)
            {
                if ((DateTime)dtAntibodyAudit.Rows[i][ARTIFICIAL.MirrorDate] < this.StartDate)
                    continue;
                if (i == dtAntibodyAudit.Rows.Count - 1)
                {
                    RowModified = false;
                }
                else
                {
                    //do not compare records of two different blood units
                    //if the record has just been added and there is no history of changes
                    if (dtAntibodyAudit.Rows[i][TABLE.AntibodyParameter.AntibodyParameterGuid].ToString() != dtAntibodyAudit.Rows[i + 1][TABLE.AntibodyParameter.AntibodyParameterGuid].ToString())
                    {
                        RowModified = false;
                    }
                    else
                    {
                        RowModified = true;
                    }
                }

                foreach (DataColumn dtCol in dtAntibodyAudit.Columns)
                {
                    if (dtCol.ColumnName == TABLE.AntibodyParameter.HigherLevelOverrideIndicator ||
                        dtCol.ColumnName == TABLE.AntibodyParameter.CompatibilityPercentage)
                    {
                        //skip columns that have not changed the value
                        if (RowModified == true)
                        {
                            if (dtAntibodyAudit.Rows[i][dtCol].Equals(dtAntibodyAudit.Rows[i + 1][dtCol]) ||
                                (dtAntibodyAudit.Rows[i].IsNull(dtCol) && dtAntibodyAudit.Rows[i + 1].IsNull(dtCol)))
                            {
                                continue;
                            }
                        }

                        DataRow drAuditReport = this._dtAuditReport.NewRow();
                        drAuditReport[ARTIFICIAL.AuditType] = AuditTrailConst.AuditTrailType.Antibody;
                        drAuditReport[ARTIFICIAL.AuditIdentifier] = dtAntibodyAudit.Rows[i][TABLE.AntibodyType.AntibodyTypeName].ToString().Trim();

                        if (dtCol.ColumnName == TABLE.AntibodyParameter.CompatibilityPercentage)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Compatibility Percentage";
                        }
                        else if (dtCol.ColumnName == TABLE.AntibodyParameter.HigherLevelOverrideIndicator)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Higher level override required indicator";
                        }

                        //previous value
                        if (RowModified == false)
                        {
                            drAuditReport[ARTIFICIAL.OldValue] = "None";
                        }
                        else
                        {
                            if (dtCol.ColumnName == TABLE.AntibodyParameter.HigherLevelOverrideIndicator)
                            {
                                if ((bool)dtAntibodyAudit.Rows[i + 1][dtCol] == true)
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "Yes";
                                }
                                else
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "No";
                                }
                            }
                            else if (dtCol.ColumnName == TABLE.AntibodyParameter.CompatibilityPercentage)
                            {
                                if (dtAntibodyAudit.Rows[i + 1].IsNull(dtCol) == false)
                                    drAuditReport[ARTIFICIAL.OldValue] = dtAntibodyAudit.Rows[i + 1][dtCol];
                                else
                                    drAuditReport[ARTIFICIAL.OldValue] = 0;
                            }
                        }

                        //new value
                        if (dtCol.ColumnName == TABLE.AntibodyParameter.HigherLevelOverrideIndicator)
                        {
                            if ((bool)dtAntibodyAudit.Rows[i][dtCol] == true)
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "Yes";
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "No";
                            }
                        }
                        else if (dtCol.ColumnName == TABLE.AntibodyParameter.CompatibilityPercentage)
                        {
                            if (dtAntibodyAudit.Rows[i].IsNull(dtCol) == false)
                                drAuditReport[ARTIFICIAL.NewValue] = dtAntibodyAudit.Rows[i][dtCol];
                            else
                                drAuditReport[ARTIFICIAL.NewValue] = 0;
                        }

                        try
                        {
                            BOL.VbecsUser LastUpdateUser = new BOL.VbecsUser(dtAntibodyAudit.Rows[i][TABLE.AntibodyParameter.LastUpdateUser].ToString().Trim());
                            if (LastUpdateUser.UserName == null)
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = dtAntibodyAudit.Rows[i][TABLE.AntibodyParameter.LastUpdateUser];
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = LastUpdateUser.UserName;
                            }
                        }
                        catch
                        {
                            drAuditReport[ARTIFICIAL.AuditUser] = dtAntibodyAudit.Rows[i][TABLE.AntibodyParameter.LastUpdateUser];
                        }


                        drAuditReport[ARTIFICIAL.AuditDate] = dtAntibodyAudit.Rows[i][ARTIFICIAL.MirrorDate];
                        drAuditReport[ARTIFICIAL.Comment] = "";

                        this._dtAuditReport.Rows.Add(drAuditReport);
                    }
                }
            }
        }

        /// <summary>
        /// Generates the DataTable used as a source for Workload Audit Report
        /// </summary>
        private void GetWorkloadAuditReport()
        {
            bool RowModified = false;
            DataTable dtWorkloadAudit = DAL.AuditTrail.GetWorkloadAudit(this.StartDate, this.EndDate);
            int nCount = 0;
            //generating one audit report row for each changed field
            nCount = dtWorkloadAudit.Rows.Count;
            for (int i = 0; i < nCount; i++)
            {
                if ((DateTime)dtWorkloadAudit.Rows[i][ARTIFICIAL.MirrorDate] < this.StartDate)
                    continue;
                if (i == dtWorkloadAudit.Rows.Count - 1)
                {
                    RowModified = false;
                }
                else
                {
                    //do not compare records of two different blood units
                    //if the record has just been added and there is no history of changes
                    if (dtWorkloadAudit.Rows[i][TABLE.WorkloadProcess.WorkloadProcessGuid].ToString() != dtWorkloadAudit.Rows[i + 1][TABLE.WorkloadProcess.WorkloadProcessGuid].ToString())
                    {
                        RowModified = false;
                    }
                    else
                    {
                        RowModified = true;
                    }
                }

                foreach (DataColumn dtCol in dtWorkloadAudit.Columns)
                {
                    if (dtCol.ColumnName == TABLE.WorkloadProcess.AssignedDate ||
                        dtCol.ColumnName == TABLE.WorkloadProcess.UnAssignedDate ||
                        dtCol.ColumnName == TABLE.WorkloadProcess.RecordStatusCode)
                    {
                        //skip columns that have not changed the value
                        if (RowModified == true)
                        {
                            if (dtWorkloadAudit.Rows[i][dtCol].Equals(dtWorkloadAudit.Rows[i + 1][dtCol]) ||
                                (dtWorkloadAudit.Rows[i].IsNull(dtCol) == true && dtWorkloadAudit.Rows[i + 1].IsNull(dtCol) == true))
                            {
                                continue;
                            }
                        }
                        else
                        {
                            //if this is a new row and the new value for column is null - skip this column
                            if (dtWorkloadAudit.Rows[i].IsNull(dtCol) == true)
                                continue;
                        }

                        DataRow drAuditReport = this._dtAuditReport.NewRow();
                        drAuditReport[ARTIFICIAL.AuditType] = AuditTrailConst.AuditTrailType.Workload;
                        drAuditReport[ARTIFICIAL.AuditIdentifier] = dtWorkloadAudit.Rows[i][TABLE.VbecsProcess.VbecsProcessName].ToString().Trim();

                        string code = dtWorkloadAudit.Rows[i][TABLE.WorkloadProcess.WorkloadCode].ToString().Trim();
                        string procedure = dtWorkloadAudit.Rows[i][TABLE.WorkloadProcess.WorkloadProcedure].ToString().Trim();

                        if (dtCol.ColumnName == TABLE.WorkloadProcess.AssignedDate)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = procedure + " (" + code + ") Start Date";
                        }
                        else if (dtCol.ColumnName == TABLE.WorkloadProcess.UnAssignedDate)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = procedure + " (" + code + ") Stop Date";
                        }
                        else if (dtCol.ColumnName == TABLE.WorkloadProcess.RecordStatusCode)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = procedure + " (" + code + ") Status Indicator";
                        }

                        //previous value
                        if (RowModified == false)
                        {
                            drAuditReport[ARTIFICIAL.OldValue] = "None";
                        }
                        else
                        {
                            if (dtCol.ColumnName == TABLE.WorkloadProcess.AssignedDate ||
                                dtCol.ColumnName == TABLE.WorkloadProcess.UnAssignedDate)
                            {
                                if (dtWorkloadAudit.Rows[i + 1].IsNull(dtCol) == false)
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = Common.VBECSDateTime.FormatDateString((DateTime)dtWorkloadAudit.Rows[i + 1][dtCol]);
                                }
                                else
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "None";
                                }
                            }
                            else if (dtCol.ColumnName == TABLE.WorkloadProcess.RecordStatusCode)
                            {
                                if (dtWorkloadAudit.Rows[i + 1][dtCol].ToString().Trim() == Convert.ToString(Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Inactive)))
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "Inactive";
                                }
                                else
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "Active";
                                }
                            }
                        }

                        //new value
                        if (dtCol.ColumnName == TABLE.WorkloadProcess.AssignedDate ||
                            dtCol.ColumnName == TABLE.WorkloadProcess.UnAssignedDate)
                        {
                            if (dtWorkloadAudit.Rows[i].IsNull(dtCol) == false)
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = Common.VBECSDateTime.FormatDateString((DateTime)dtWorkloadAudit.Rows[i][dtCol]);
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "None";
                            }
                        }
                        else if (dtCol.ColumnName == TABLE.WorkloadProcess.RecordStatusCode)
                        {
                            if (dtWorkloadAudit.Rows[i][dtCol].ToString().Trim() == Convert.ToString(Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Inactive)))
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "Inactive";
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "Active";
                            }
                        }

                        try
                        {
                            BOL.VbecsUser LastUpdateUser = new BOL.VbecsUser(dtWorkloadAudit.Rows[i][TABLE.SpecimenTestThreshold.LastUpdateUser].ToString().Trim());
                            if (LastUpdateUser.UserName == null)
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = dtWorkloadAudit.Rows[i][TABLE.SpecimenTestThreshold.LastUpdateUser];
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = LastUpdateUser.UserName;
                            }
                        }
                        catch
                        {
                            drAuditReport[ARTIFICIAL.AuditUser] = dtWorkloadAudit.Rows[i][TABLE.SpecimenTestThreshold.LastUpdateUser];
                        }

                        drAuditReport[ARTIFICIAL.AuditDate] = dtWorkloadAudit.Rows[i][ARTIFICIAL.MirrorDate];
                        drAuditReport[ARTIFICIAL.Comment] = "";

                        this._dtAuditReport.Rows.Add(drAuditReport);
                    }
                }
            }
        }

        /// <summary>
        /// Generates the DataTable used as a source for MSBOS Audit Report
        /// </summary>
        private void GetMSBOSAuditReport()
        {
            bool RowModified = false;
            DataTable dtMSBOSAudit = DAL.AuditTrail.GetMsbosAudit(this.StartDate, this.EndDate);
            int nCount = 0;
            //generating one audit report row for each changed field
            nCount = dtMSBOSAudit.Rows.Count;
            for (int i = 0; i < nCount; i++)
            {
                if ((DateTime)dtMSBOSAudit.Rows[i][ARTIFICIAL.MirrorDate] < this.StartDate)
                    continue;
                if (i == dtMSBOSAudit.Rows.Count - 1)
                {
                    RowModified = false;
                }
                else
                {
                    //do not compare records of two different blood units
                    //if the record has just been added and there is no history of changes
                    if (dtMSBOSAudit.Rows[i][TABLE.Msbos.MsbosGuid].ToString() != dtMSBOSAudit.Rows[i + 1][TABLE.Msbos.MsbosGuid].ToString())
                    {
                        RowModified = false;
                    }
                    else
                    {
                        RowModified = true;
                    }
                }

                foreach (DataColumn dtCol in dtMSBOSAudit.Columns)
                {
                    if (dtCol.ColumnName == TABLE.Msbos.TypeAndScreenIndicator ||
                        dtCol.ColumnName == TABLE.Msbos.NoBloodRequiredIndicator ||
                        dtCol.ColumnName == TABLE.Msbos.RecordStatusCode)
                    {
                        //skip columns that have not changed the value
                        if (RowModified == true)
                        {
                            if (dtMSBOSAudit.Rows[i][dtCol].Equals(dtMSBOSAudit.Rows[i + 1][dtCol]))
                            {
                                continue;
                            }
                        }

                        DataRow drAuditReport = this._dtAuditReport.NewRow();
                        drAuditReport[ARTIFICIAL.AuditType] = AuditTrailConst.AuditTrailType.MSBOS;
                        drAuditReport[ARTIFICIAL.AuditIdentifier] = dtMSBOSAudit.Rows[i][TABLE.Msbos.SurgeryName].ToString().Trim();

                        if (dtCol.ColumnName == TABLE.Msbos.TypeAndScreenIndicator)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "TAS Indicator";
                        }
                        else if (dtCol.ColumnName == TABLE.Msbos.NoBloodRequiredIndicator)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "No blood required indicator";
                        }
                        else if (dtCol.ColumnName == TABLE.Msbos.RecordStatusCode)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Surgery Status Indicator";
                        }

                        //previous value
                        if (RowModified == false)
                        {
                            drAuditReport[ARTIFICIAL.OldValue] = "None";
                        }
                        else
                        {
                            if (dtCol.ColumnName == TABLE.Msbos.TypeAndScreenIndicator ||
                                dtCol.ColumnName == TABLE.Msbos.NoBloodRequiredIndicator)
                            {
                                if ((bool)dtMSBOSAudit.Rows[i + 1][dtCol] == true)
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "Enabled";
                                }
                                else
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "Disabled";
                                }
                            }
                            else if (dtCol.ColumnName == TABLE.Msbos.RecordStatusCode)
                            {
                                if (dtMSBOSAudit.Rows[i + 1][dtCol].ToString().Trim() == Convert.ToString(Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Inactive)))
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "Inactive";
                                }
                                else
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "Active";
                                }
                            }
                            else if (dtCol.ColumnName == TABLE.Msbos.SurgeryName)
                            {
                                drAuditReport[ARTIFICIAL.OldValue] = dtMSBOSAudit.Rows[i + 1][dtCol];
                            }
                        }

                        //new value
                        if (dtCol.ColumnName == TABLE.Msbos.TypeAndScreenIndicator ||
                            dtCol.ColumnName == TABLE.Msbos.NoBloodRequiredIndicator)
                        {
                            if ((bool)dtMSBOSAudit.Rows[i][dtCol] == true)
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "Enabled";
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "Disabled";
                            }
                        }
                        else if (dtCol.ColumnName == TABLE.Msbos.RecordStatusCode)
                        {
                            if (dtMSBOSAudit.Rows[i][dtCol].ToString().Trim() == Convert.ToString(Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Inactive)))
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "Inactive";
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "Active";
                            }
                        }
                        else if (dtCol.ColumnName == TABLE.Msbos.SurgeryName)
                        {
                            drAuditReport[ARTIFICIAL.NewValue] = dtMSBOSAudit.Rows[i][dtCol];
                        }

                        try
                        {
                            BOL.VbecsUser LastUpdateUser = new BOL.VbecsUser(dtMSBOSAudit.Rows[i][TABLE.SpecimenTestThreshold.LastUpdateUser].ToString().Trim());
                            if (LastUpdateUser.UserName == null)
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = dtMSBOSAudit.Rows[i][TABLE.SpecimenTestThreshold.LastUpdateUser];
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = LastUpdateUser.UserName;
                            }
                        }
                        catch
                        {
                            drAuditReport[ARTIFICIAL.AuditUser] = dtMSBOSAudit.Rows[i][TABLE.SpecimenTestThreshold.LastUpdateUser];
                        }

                        drAuditReport[ARTIFICIAL.AuditDate] = dtMSBOSAudit.Rows[i][ARTIFICIAL.MirrorDate];
                        drAuditReport[ARTIFICIAL.Comment] = "";

                        this._dtAuditReport.Rows.Add(drAuditReport);
                    }
                }
            }

            //MSBOS Component Class
            DataTable dtMSBOSComponentClassAudit = DAL.AuditTrail.GetMsbosComponentClassAudit(this.StartDate, this.EndDate);
            nCount = dtMSBOSComponentClassAudit.Rows.Count;
            //generating one audit report row for each changed field
            for (int i = 0; i < nCount; i++)
            {
                if ((DateTime)dtMSBOSComponentClassAudit.Rows[i][ARTIFICIAL.MirrorDate] < this.StartDate)
                    continue;
                if (i == dtMSBOSComponentClassAudit.Rows.Count - 1)
                {
                    RowModified = false;
                }
                else
                {
                    //do not compare records of two different component classes
                    //if the record has just been added and there is no history of changes
                    if (dtMSBOSComponentClassAudit.Rows[i][TABLE.MsbosComponentClass.MsbosGuid].ToString() != dtMSBOSComponentClassAudit.Rows[i + 1][TABLE.MsbosComponentClass.MsbosGuid].ToString() ||
                        dtMSBOSComponentClassAudit.Rows[i][TABLE.ComponentClass.ComponentClassShortName].ToString() != dtMSBOSComponentClassAudit.Rows[i + 1][TABLE.ComponentClass.ComponentClassShortName].ToString())
                    {
                        RowModified = false;
                    }
                    else
                    {
                        RowModified = true;
                    }
                }

                foreach (DataColumn dtCol in dtMSBOSComponentClassAudit.Columns)
                {
                    if (dtCol.ColumnName == TABLE.MsbosComponentClass.MaximumSetupUnitQuantity ||
                        dtCol.ColumnName == TABLE.MsbosComponentClass.RecordStatusCode)
                    {
                        //skip columns that have not changed the value
                        if (RowModified == true)
                        {
                            if (dtMSBOSComponentClassAudit.Rows[i][dtCol].Equals(dtMSBOSComponentClassAudit.Rows[i + 1][dtCol]))
                            {
                                continue;
                            }
                        }

                        DataRow drAuditReport = this._dtAuditReport.NewRow();
                        drAuditReport[ARTIFICIAL.AuditType] = AuditTrailConst.AuditTrailType.MSBOS;
                        drAuditReport[ARTIFICIAL.AuditIdentifier] = dtMSBOSComponentClassAudit.Rows[i][TABLE.Msbos.SurgeryName].ToString().Trim();

                        if (dtCol.ColumnName == TABLE.MsbosComponentClass.MaximumSetupUnitQuantity)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Number of units (" + dtMSBOSComponentClassAudit.Rows[i][TABLE.ComponentClass.ComponentClassShortName].ToString().Trim() + ")";
                        }
                        else if (dtCol.ColumnName == TABLE.Msbos.RecordStatusCode)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Recommendation Status Indicator (" + dtMSBOSComponentClassAudit.Rows[i][TABLE.ComponentClass.ComponentClassShortName].ToString().Trim() + ")";
                        }

                        //previous value
                        if (RowModified == false)
                        {
                            drAuditReport[ARTIFICIAL.OldValue] = "None";
                        }
                        else
                        {
                            if (dtCol.ColumnName == TABLE.Msbos.RecordStatusCode)
                            {
                                if (dtMSBOSComponentClassAudit.Rows[i + 1][dtCol].ToString().Trim() == Convert.ToString(Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Inactive)))
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "Inactive";
                                }
                                else
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "Active";
                                }
                            }
                            else if (dtCol.ColumnName == TABLE.MsbosComponentClass.MaximumSetupUnitQuantity)
                            {
                                drAuditReport[ARTIFICIAL.OldValue] = dtMSBOSComponentClassAudit.Rows[i + 1][dtCol];
                            }
                        }

                        //new value
                        if (dtCol.ColumnName == TABLE.Msbos.RecordStatusCode)
                        {
                            if (dtMSBOSComponentClassAudit.Rows[i][dtCol].ToString().Trim() == Convert.ToString(Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Inactive)))
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "Inactive";
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "Active";
                            }
                        }
                        else if (dtCol.ColumnName == TABLE.MsbosComponentClass.MaximumSetupUnitQuantity)
                        {
                            drAuditReport[ARTIFICIAL.NewValue] = dtMSBOSComponentClassAudit.Rows[i][dtCol];
                        }

                        try
                        {
                            BOL.VbecsUser LastUpdateUser = new BOL.VbecsUser(dtMSBOSComponentClassAudit.Rows[i][TABLE.SpecimenTestThreshold.LastUpdateUser].ToString().Trim());
                            if (LastUpdateUser.UserName == null)
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = dtMSBOSComponentClassAudit.Rows[i][TABLE.SpecimenTestThreshold.LastUpdateUser];
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = LastUpdateUser.UserName;
                            }
                        }
                        catch
                        {
                            drAuditReport[ARTIFICIAL.AuditUser] = dtMSBOSComponentClassAudit.Rows[i][TABLE.SpecimenTestThreshold.LastUpdateUser];
                        }

                        drAuditReport[ARTIFICIAL.AuditDate] = dtMSBOSComponentClassAudit.Rows[i][ARTIFICIAL.MirrorDate];
                        drAuditReport[ARTIFICIAL.Comment] = "";

                        this._dtAuditReport.Rows.Add(drAuditReport);
                    }
                }
            }
        }

        /// <summary>
        /// Generates the DataTable used as a source for Transfusion Effectiveness Report
        /// </summary>
        private void GetTransfusionEffectivenessAuditReport()
        {
            bool RowModified = false;
            DataTable dtTransfusionEffectivenessAudit = DAL.AuditTrail.GetTransfusionEffectivenessAudit(this.StartDate, this.EndDate);
            int nCount = 0;
            //generating one audit report row for each changed field
            nCount = dtTransfusionEffectivenessAudit.Rows.Count;
            for (int i = 0; i < nCount; i++)
            {
                if ((DateTime)dtTransfusionEffectivenessAudit.Rows[i][ARTIFICIAL.MirrorDate] < this.StartDate)
                    continue;
                if (i == dtTransfusionEffectivenessAudit.Rows.Count - 1)
                {
                    RowModified = false;
                }
                else
                {
                    //do not compare records of two different blood units
                    //if the record has just been added and there is no history of changes
                    if (dtTransfusionEffectivenessAudit.Rows[i][TABLE.SpecimenTestThreshold.SpecimenTestThresholdGuid].ToString() != dtTransfusionEffectivenessAudit.Rows[i + 1][TABLE.SpecimenTestThreshold.SpecimenTestThresholdGuid].ToString())
                    {
                        RowModified = false;
                    }
                    else
                    {
                        RowModified = true;
                    }
                }

                foreach (DataColumn dtCol in dtTransfusionEffectivenessAudit.Columns)
                {
                    if (dtCol.ColumnName == TABLE.SpecimenTestThreshold.RecordStatusCode)
                    {
                        //skip columns that have not changed the value
                        if (RowModified == true)
                        {
                            if (dtTransfusionEffectivenessAudit.Rows[i][dtCol].Equals(dtTransfusionEffectivenessAudit.Rows[i + 1][dtCol]) ||
                                (dtTransfusionEffectivenessAudit.Rows[i].IsNull(dtCol) && dtTransfusionEffectivenessAudit.Rows[i + 1].IsNull(dtCol)) ||
                                (dtTransfusionEffectivenessAudit.Rows[i].IsNull(dtCol) && dtTransfusionEffectivenessAudit.Rows[i + 1][dtCol].ToString().Trim() == "") ||
                                (dtTransfusionEffectivenessAudit.Rows[i][dtCol].ToString().Trim() == "" && dtTransfusionEffectivenessAudit.Rows[i + 1][dtCol].ToString().Trim() == ""))
                            {
                                continue;
                            }
                        }
                        else
                        {
                            if (dtTransfusionEffectivenessAudit.Rows[i].IsNull(dtCol))
                                continue;
                        }

                        DataRow drAuditReport = this._dtAuditReport.NewRow();
                        drAuditReport[ARTIFICIAL.AuditType] = AuditTrailConst.AuditTrailType.TransfusionEffectiveness;
                        drAuditReport[ARTIFICIAL.AuditIdentifier] = dtTransfusionEffectivenessAudit.Rows[i][TABLE.SpecimenTestThreshold.VistaLaboratoryTestName].ToString().Trim() + " (IEN: " + dtTransfusionEffectivenessAudit.Rows[i][TABLE.SpecimenTestThreshold.LabTestId].ToString().Trim() + ", Specimen Type: " + dtTransfusionEffectivenessAudit.Rows[i][TABLE.SpecimenType.SpecimenTypeText].ToString().Trim() + ")";

                        if (dtCol.ColumnName == TABLE.SpecimenTestThreshold.RecordStatusCode)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Status Indicator";
                        }

                        //previous value
                        if (RowModified == false)
                        {
                            drAuditReport[ARTIFICIAL.OldValue] = "None";
                        }
                        else
                        {
                            if (dtCol.ColumnName == TABLE.SpecimenTestThreshold.RecordStatusCode)
                            {
                                if (dtTransfusionEffectivenessAudit.Rows[i + 1][dtCol].ToString().Trim() == Convert.ToString(Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active)))
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "Active";
                                }
                                else
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "Inactive";
                                }
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.OldValue] = dtTransfusionEffectivenessAudit.Rows[i + 1][dtCol].ToString().Trim();
                            }
                        }
                        if (drAuditReport[ARTIFICIAL.OldValue].ToString().Trim() == "")
                            drAuditReport[ARTIFICIAL.OldValue] = "None";

                        //new value
                        if (dtCol.ColumnName == TABLE.SpecimenTestThreshold.RecordStatusCode)
                        {
                            if (dtTransfusionEffectivenessAudit.Rows[i][dtCol].ToString().Trim() == Convert.ToString(Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active)))
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "Active";
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "Inactive";
                            }
                        }
                        else
                        {
                            drAuditReport[ARTIFICIAL.NewValue] = dtTransfusionEffectivenessAudit.Rows[i][dtCol].ToString().Trim();
                        }
                        if (drAuditReport[ARTIFICIAL.NewValue].ToString().Trim() == "")
                            drAuditReport[ARTIFICIAL.NewValue] = "None";

                        try
                        {
                            BOL.VbecsUser LastUpdateUser = new BOL.VbecsUser(dtTransfusionEffectivenessAudit.Rows[i][TABLE.VamcDivision.LastUpdateUser].ToString().Trim());
                            if (LastUpdateUser.UserName == null)
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = dtTransfusionEffectivenessAudit.Rows[i][TABLE.VamcDivision.LastUpdateUser];
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = LastUpdateUser.UserName;
                            }
                        }
                        catch
                        {
                            drAuditReport[ARTIFICIAL.AuditUser] = dtTransfusionEffectivenessAudit.Rows[i][TABLE.VamcDivision.LastUpdateUser];
                        }

                        drAuditReport[ARTIFICIAL.AuditDate] = dtTransfusionEffectivenessAudit.Rows[i][ARTIFICIAL.MirrorDate];
                        drAuditReport[ARTIFICIAL.Comment] = "";

                        this._dtAuditReport.Rows.Add(drAuditReport);
                    }
                }
            }
        }

        /// <summary>
        /// Generates the DataTable used as a source for Transfusion Complications Report
        /// </summary>
        private void GetTransfusionComplicationsAuditReport()
        {
            bool RowModified = false;
            DataTable dtTransfusionComplicationsAudit = DAL.AuditTrail.GetTransfusionComplicationsAudit(this.StartDate, this.EndDate);
            int nCount = 0;
            //adding threshold type to the table

            dtTransfusionComplicationsAudit.Columns.Add(ARTIFICIAL.ThresholdType);
            nCount = dtTransfusionComplicationsAudit.Rows.Count;
            for (int i = 0; i < nCount; i++)
            {
                if (Common.RegularExpressions.ThresholdText().IsMatch(dtTransfusionComplicationsAudit.Rows[i][TABLE.SpecimenTestThreshold.ThresholdResult].ToString()))
                {
                    dtTransfusionComplicationsAudit.Rows[i][ARTIFICIAL.ThresholdType] = "Positive Result(s)";
                }
                else if (Common.RegularExpressions.Threshold().IsMatch(dtTransfusionComplicationsAudit.Rows[i][TABLE.SpecimenTestThreshold.ThresholdResult].ToString()))
                {
                    dtTransfusionComplicationsAudit.Rows[i][ARTIFICIAL.ThresholdType] = "Numeric Threshold";
                }
                else
                {
                    dtTransfusionComplicationsAudit.Rows[i][ARTIFICIAL.ThresholdType] = "No Threshold";
                }
            }

            //generating one audit report row for each changed field
            nCount = dtTransfusionComplicationsAudit.Rows.Count;
            for (int i = 0; i < nCount; i++)
            {
                if ((DateTime)dtTransfusionComplicationsAudit.Rows[i][ARTIFICIAL.MirrorDate] < this.StartDate)
                    continue;
                if (i == dtTransfusionComplicationsAudit.Rows.Count - 1)
                {
                    RowModified = false;
                }
                else
                {
                    //do not compare records of two different blood units
                    //if the record has just been added and there is no history of changes
                    if (dtTransfusionComplicationsAudit.Rows[i][TABLE.SpecimenTestThreshold.SpecimenTestThresholdGuid].ToString() != dtTransfusionComplicationsAudit.Rows[i + 1][TABLE.SpecimenTestThreshold.SpecimenTestThresholdGuid].ToString())
                    {
                        RowModified = false;
                    }
                    else
                    {
                        RowModified = true;
                    }
                }

                foreach (DataColumn dtCol in dtTransfusionComplicationsAudit.Columns)
                {
                    if (dtCol.ColumnName == TABLE.SpecimenTestThreshold.RecordStatusCode ||
                        dtCol.ColumnName == ARTIFICIAL.ThresholdType ||
                        dtCol.ColumnName == TABLE.SpecimenTestThreshold.ThresholdResult)
                    {
                        //skip columns that have not changed the value
                        if (RowModified == true)
                        {
                            if (dtTransfusionComplicationsAudit.Rows[i][dtCol].Equals(dtTransfusionComplicationsAudit.Rows[i + 1][dtCol]) ||
                                (dtTransfusionComplicationsAudit.Rows[i].IsNull(dtCol) && dtTransfusionComplicationsAudit.Rows[i + 1].IsNull(dtCol)) ||
                                (dtTransfusionComplicationsAudit.Rows[i].IsNull(dtCol) && dtTransfusionComplicationsAudit.Rows[i + 1][dtCol].ToString().Trim() == "") ||
                                (dtTransfusionComplicationsAudit.Rows[i][dtCol].ToString().Trim() == "" && dtTransfusionComplicationsAudit.Rows[i + 1][dtCol].ToString().Trim() == ""))
                            {
                                continue;
                            }
                        }
                        else
                        {
                            if (dtTransfusionComplicationsAudit.Rows[i].IsNull(dtCol))
                                continue;
                        }

                        DataRow drAuditReport = this._dtAuditReport.NewRow();
                        drAuditReport[ARTIFICIAL.AuditType] = AuditTrailConst.AuditTrailType.TransfusionComplications;
                        drAuditReport[ARTIFICIAL.AuditIdentifier] = dtTransfusionComplicationsAudit.Rows[i][TABLE.SpecimenTestThreshold.VistaLaboratoryTestName].ToString().Trim() + " (IEN: " + dtTransfusionComplicationsAudit.Rows[i][TABLE.SpecimenTestThreshold.LabTestId].ToString().Trim() + ", Specimen Type: " + dtTransfusionComplicationsAudit.Rows[i][TABLE.SpecimenType.SpecimenTypeText].ToString().Trim() + ")";

                        if (dtCol.ColumnName == TABLE.SpecimenTestThreshold.RecordStatusCode)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Status Indicator";
                        }
                        else if (dtCol.ColumnName == TABLE.SpecimenTestThreshold.ThresholdResult)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Threshold Result";
                        }
                        else if (dtCol.ColumnName == ARTIFICIAL.ThresholdType)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Threshold Setting Status";
                        }


                        //previous value
                        if (RowModified == false)
                        {
                            drAuditReport[ARTIFICIAL.OldValue] = "None";
                        }
                        else
                        {
                            if (dtCol.ColumnName == TABLE.SpecimenTestThreshold.RecordStatusCode)
                            {
                                if (dtTransfusionComplicationsAudit.Rows[i + 1][dtCol].ToString().Trim() == Convert.ToString(Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active)))
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "Active";
                                }
                                else
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "Inactive";
                                }
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.OldValue] = dtTransfusionComplicationsAudit.Rows[i + 1][dtCol].ToString().Trim();
                            }
                        }
                        if (drAuditReport[ARTIFICIAL.OldValue].ToString().Trim() == "")
                            drAuditReport[ARTIFICIAL.OldValue] = "None";

                        //new value
                        if (dtCol.ColumnName == TABLE.SpecimenTestThreshold.RecordStatusCode)
                        {
                            if (dtTransfusionComplicationsAudit.Rows[i][dtCol].ToString().Trim() == Convert.ToString(Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active)))
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "Active";
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "Inactive";
                            }
                        }
                        else
                        {
                            drAuditReport[ARTIFICIAL.NewValue] = dtTransfusionComplicationsAudit.Rows[i][dtCol].ToString().Trim();
                        }
                        if (drAuditReport[ARTIFICIAL.NewValue].ToString().Trim() == "")
                            drAuditReport[ARTIFICIAL.NewValue] = "None";

                        try
                        {
                            BOL.VbecsUser LastUpdateUser = new BOL.VbecsUser(dtTransfusionComplicationsAudit.Rows[i][TABLE.VamcDivision.LastUpdateUser].ToString().Trim());
                            if (LastUpdateUser.UserName == null)
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = dtTransfusionComplicationsAudit.Rows[i][TABLE.VamcDivision.LastUpdateUser];
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = LastUpdateUser.UserName;
                            }
                        }
                        catch
                        {
                            drAuditReport[ARTIFICIAL.AuditUser] = dtTransfusionComplicationsAudit.Rows[i][TABLE.VamcDivision.LastUpdateUser];
                        }

                        drAuditReport[ARTIFICIAL.AuditDate] = dtTransfusionComplicationsAudit.Rows[i][ARTIFICIAL.MirrorDate];
                        drAuditReport[ARTIFICIAL.Comment] = "";

                        this._dtAuditReport.Rows.Add(drAuditReport);
                    }
                }
            }
        }

        /// <summary>
        /// Generates the DataTable used as a source for Component Class Audit Report
        /// </summary>
        private void GetComponentClassAuditReport()
        {
            bool RowModified = false;
            DataTable dtComponentClassAudit = DAL.AuditTrail.GetComponentClassAudit(this.StartDate, this.EndDate);
            int nCount = 0;
            //generating one audit report row for each changed field
            nCount = dtComponentClassAudit.Rows.Count;
            for (int i = 0; i < nCount; i++)
            {
                if ((DateTime)dtComponentClassAudit.Rows[i][ARTIFICIAL.MirrorDate] < this.StartDate)
                    continue;
                if (i == dtComponentClassAudit.Rows.Count - 1)
                {
                    RowModified = false;
                }
                else
                {
                    //do not compare records of two different blood units
                    //if the record has just been added and there is no history of changes
                    if (dtComponentClassAudit.Rows[i][TABLE.ComponentClassParameter.ComponentClassParameterGuid].ToString() != dtComponentClassAudit.Rows[i + 1][TABLE.ComponentClassParameter.ComponentClassParameterGuid].ToString())
                    {
                        RowModified = false;
                    }
                    else
                    {
                        RowModified = true;
                    }
                }

                foreach (DataColumn dtCol in dtComponentClassAudit.Columns)
                {
                    if (dtCol.ColumnName == TABLE.ComponentClassParameter.SpecimenRequiredIndicator ||
                        dtCol.ColumnName == TABLE.ComponentClassParameter.MaximumTransfusionTime)
                    {
                        //skip columns that have not changed the value
                        if (RowModified == true)
                        {
                            if (dtComponentClassAudit.Rows[i][dtCol].Equals(dtComponentClassAudit.Rows[i + 1][dtCol]))
                            {
                                continue;
                            }
                        }

                        DataRow drAuditReport = this._dtAuditReport.NewRow();
                        drAuditReport[ARTIFICIAL.AuditType] = AuditTrailConst.AuditTrailType.ComponentClass;
                        drAuditReport[ARTIFICIAL.AuditIdentifier] = dtComponentClassAudit.Rows[i][TABLE.ComponentClass.ComponentClassName].ToString().Trim();

                        if (dtCol.ColumnName == TABLE.ComponentClassParameter.SpecimenRequiredIndicator)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Current ABO/Rh specimen results required indicator";
                        }
                        else if (dtCol.ColumnName == TABLE.ComponentClassParameter.MaximumTransfusionTime)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Maximum transfusion time";
                        }

                        //previous value
                        if (RowModified == false)
                        {
                            drAuditReport[ARTIFICIAL.OldValue] = "None";
                        }
                        else
                        {
                            if (dtCol.ColumnName == TABLE.ComponentClassParameter.SpecimenRequiredIndicator)
                            {
                                if ((bool)dtComponentClassAudit.Rows[i + 1][dtCol] == true)
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "Yes";
                                }
                                else
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "No";
                                }
                            }
                            else if (dtCol.ColumnName == TABLE.ComponentClassParameter.MaximumTransfusionTime)
                            {
                                drAuditReport[ARTIFICIAL.OldValue] = dtComponentClassAudit.Rows[i + 1][dtCol];
                            }
                        }

                        //new value
                        if (dtCol.ColumnName == TABLE.ComponentClassParameter.SpecimenRequiredIndicator)
                        {
                            if ((bool)dtComponentClassAudit.Rows[i][dtCol] == true)
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "Yes";
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "No";
                            }
                        }
                        else if (dtCol.ColumnName == TABLE.ComponentClassParameter.MaximumTransfusionTime)
                        {
                            drAuditReport[ARTIFICIAL.NewValue] = dtComponentClassAudit.Rows[i][dtCol];
                        }

                        try
                        {
                            BOL.VbecsUser LastUpdateUser = new BOL.VbecsUser(dtComponentClassAudit.Rows[i][TABLE.SpecimenTestThreshold.LastUpdateUser].ToString().Trim());
                            if (LastUpdateUser.UserName == null)
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = dtComponentClassAudit.Rows[i][TABLE.SpecimenTestThreshold.LastUpdateUser];
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = LastUpdateUser.UserName;
                            }
                        }
                        catch
                        {
                            drAuditReport[ARTIFICIAL.AuditUser] = dtComponentClassAudit.Rows[i][TABLE.SpecimenTestThreshold.LastUpdateUser];
                        }

                        drAuditReport[ARTIFICIAL.AuditDate] = dtComponentClassAudit.Rows[i][ARTIFICIAL.MirrorDate];
                        drAuditReport[ARTIFICIAL.Comment] = "";

                        this._dtAuditReport.Rows.Add(drAuditReport);
                    }
                }
            }

            DataTable dtInappropriateRequestAudit = DAL.AuditTrail.GetInappropriateRequestIndicatorsAudit(this.StartDate, this.EndDate);

            //generating one audit report row for each changed field
            nCount = dtInappropriateRequestAudit.Rows.Count;
            for (int i = 0; i < nCount; i++)
            {
                if ((DateTime)dtInappropriateRequestAudit.Rows[i][ARTIFICIAL.MirrorDate] < this.StartDate)
                    continue;
                if (i == dtInappropriateRequestAudit.Rows.Count - 1)
                {
                    RowModified = false;
                }
                else
                {
                    //do not compare records of two different blood units
                    //if the record has just been added and there is no history of changes
                    if (dtInappropriateRequestAudit.Rows[i][TABLE.SpecimenTestThreshold.SpecimenTestThresholdGuid].ToString() != dtInappropriateRequestAudit.Rows[i + 1][TABLE.SpecimenTestThreshold.SpecimenTestThresholdGuid].ToString())
                    {
                        RowModified = false;
                    }
                    else
                    {
                        RowModified = true;
                    }
                }

                foreach (DataColumn dtCol in dtInappropriateRequestAudit.Columns)
                {
                    if (dtCol.ColumnName == TABLE.SpecimenTestThreshold.RecordStatusCode ||
                        dtCol.ColumnName == TABLE.SpecimenTestThreshold.ThresholdResult)
                    {
                        //skip columns that have not changed the value
                        if (RowModified == true)
                        {
                            if (dtInappropriateRequestAudit.Rows[i][dtCol].Equals(dtInappropriateRequestAudit.Rows[i + 1][dtCol]))
                            {
                                continue;
                            }
                        }

                        DataRow drAuditReport = this._dtAuditReport.NewRow();
                        drAuditReport[ARTIFICIAL.AuditType] = AuditTrailConst.AuditTrailType.ComponentClass;
                        drAuditReport[ARTIFICIAL.AuditIdentifier] = dtInappropriateRequestAudit.Rows[i][TABLE.ComponentClass.ComponentClassName].ToString().Trim();

                        if (dtCol.ColumnName == TABLE.SpecimenTestThreshold.RecordStatusCode)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Lab test status for " + dtInappropriateRequestAudit.Rows[i][TABLE.SpecimenTestThreshold.VistaLaboratoryTestName].ToString().Trim() + " (IEN: " + dtInappropriateRequestAudit.Rows[i][TABLE.SpecimenTestThreshold.LabTestId].ToString().Trim() + ", Specimen Type: " + dtInappropriateRequestAudit.Rows[i][TABLE.SpecimenType.SpecimenTypeText].ToString().Trim() + ")";
                        }
                        else if (dtCol.ColumnName == TABLE.SpecimenTestThreshold.ThresholdResult)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Threshold result for " + dtInappropriateRequestAudit.Rows[i][TABLE.SpecimenTestThreshold.VistaLaboratoryTestName].ToString().Trim() + " (IEN: " + dtInappropriateRequestAudit.Rows[i][TABLE.SpecimenTestThreshold.LabTestId].ToString().Trim() + ", Specimen Type: " + dtInappropriateRequestAudit.Rows[i][TABLE.SpecimenType.SpecimenTypeText].ToString().Trim() + ")";
                        }

                        //previous value
                        if (RowModified == false)
                        {
                            drAuditReport[ARTIFICIAL.OldValue] = "None";
                        }
                        else
                        {
                            if (dtCol.ColumnName == TABLE.SpecimenTestThreshold.RecordStatusCode)
                            {
                                if (dtInappropriateRequestAudit.Rows[i + 1][dtCol].ToString().Trim() == Convert.ToString(Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Inactive)))
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "Inactive";
                                }
                                else
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "Active";
                                }
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.OldValue] = dtInappropriateRequestAudit.Rows[i + 1][dtCol].ToString().Trim();
                            }
                        }

                        //new value
                        if (dtCol.ColumnName == TABLE.SpecimenTestThreshold.RecordStatusCode)
                        {
                            if (dtInappropriateRequestAudit.Rows[i][dtCol].ToString().Trim() == Convert.ToString(Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Inactive)))
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "Inactive";
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "Active";
                            }
                        }
                        else
                        {
                            drAuditReport[ARTIFICIAL.NewValue] = dtInappropriateRequestAudit.Rows[i][dtCol].ToString().Trim();
                        }

                        try
                        {
                            BOL.VbecsUser LastUpdateUser = new BOL.VbecsUser(dtInappropriateRequestAudit.Rows[i][TABLE.SpecimenTestThreshold.LastUpdateUser].ToString().Trim());
                            if (LastUpdateUser.UserName == null)
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = dtInappropriateRequestAudit.Rows[i][TABLE.SpecimenTestThreshold.LastUpdateUser];
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = LastUpdateUser.UserName;
                            }
                        }
                        catch
                        {
                            drAuditReport[ARTIFICIAL.AuditUser] = dtInappropriateRequestAudit.Rows[i][TABLE.SpecimenTestThreshold.LastUpdateUser];
                        }

                        drAuditReport[ARTIFICIAL.AuditDate] = dtInappropriateRequestAudit.Rows[i][ARTIFICIAL.MirrorDate];
                        drAuditReport[ARTIFICIAL.Comment] = "";

                        this._dtAuditReport.Rows.Add(drAuditReport);
                    }
                }
            }
        }

        /// <summary>
        /// Generates the DataTable used as a source for Daily QC Configuration Audit Report
        /// </summary>
        private void GetDailyControlAuditReport()
        {
            bool RowModified = false;
            DataTable dtDailyControlAudit = DAL.AuditTrail.GetDailyControlAudit(this.StartDate, this.EndDate);
            int nCount = 0;
            //generating one audit report row for each changed field
            nCount = dtDailyControlAudit.Rows.Count;
            for (int i = 0; i < nCount; i++)
            {
                if ((DateTime)dtDailyControlAudit.Rows[i][ARTIFICIAL.MirrorDate] < this.StartDate)
                    continue;
                if (i == dtDailyControlAudit.Rows.Count - 1)
                {
                    RowModified = false;
                }
                else
                {
                    RowModified = true;
                }

                foreach (DataColumn dtCol in dtDailyControlAudit.Columns)
                {
                    if (dtCol.ColumnName == TABLE.DailyControl.CommercialTemplateIndicator ||
                        dtCol.ColumnName == TABLE.DailyControl.RackNamesIndicator ||
                        dtCol.ColumnName == TABLE.DailyControl.NumberOfRacks ||
                        dtCol.ColumnName == TABLE.DailyControl.DailyAlertTime ||
                        dtCol.ColumnName == TABLE.DailyControl.PrimaryEnhancementMediaId ||
                        dtCol.ColumnName == TABLE.DailyControl.SecondaryEnhancementMediaId)
                    {
                        //skip columns that have not changed the value
                        if (RowModified == true)
                        {
                            if (dtCol.ColumnName != TABLE.DailyControl.DailyAlertTime)
                            {
                                if (dtDailyControlAudit.Rows[i][dtCol].Equals(dtDailyControlAudit.Rows[i + 1][dtCol]) ||
                                    dtDailyControlAudit.Rows[i][dtCol].ToString().Trim() == dtDailyControlAudit.Rows[i + 1][dtCol].ToString().Trim())
                                {
                                    continue;
                                }
                            }
                            else
                            {
                                if (dtDailyControlAudit.Rows[i].IsNull(dtCol) == false && dtDailyControlAudit.Rows[i + 1].IsNull(dtCol) == false)
                                {
                                    if (((DateTime)dtDailyControlAudit.Rows[i][dtCol]).ToString("t").Trim() == ((DateTime)dtDailyControlAudit.Rows[i + 1][dtCol]).ToString("t").Trim())
                                    {
                                        continue;
                                    }
                                }
                                else if (dtDailyControlAudit.Rows[i].IsNull(dtCol) == true && dtDailyControlAudit.Rows[i + 1].IsNull(dtCol) == true)
                                    continue;
                            }
                        }
                        else
                        {
                            if (dtDailyControlAudit.Rows[i].IsNull(dtCol) ||
                                dtDailyControlAudit.Rows[i][dtCol].ToString().Trim() == "")
                            {
                                continue;
                            }
                        }

                        DataRow drAuditReport = this._dtAuditReport.NewRow();
                        drAuditReport[ARTIFICIAL.AuditType] = AuditTrailConst.AuditTrailType.QCConfiguration;
                        drAuditReport[ARTIFICIAL.AuditIdentifier] = AuditTrailConst.AuditIdentifier.QCSetup;

                        if (dtCol.ColumnName == TABLE.DailyControl.CommercialTemplateIndicator)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Commercial or Non-Commercial QC Indicator";
                        }
                        else if (dtCol.ColumnName == TABLE.DailyControl.RackNamesIndicator)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Rack Name Indicator";
                        }
                        else if (dtCol.ColumnName == TABLE.DailyControl.NumberOfRacks)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Number of racks to test";
                        }
                        else if (dtCol.ColumnName == TABLE.DailyControl.DailyAlertTime)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Daily Alert Time";
                        }
                        else if (dtCol.ColumnName == TABLE.DailyControl.PrimaryEnhancementMediaId)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Primary Enhancement Media";
                        }
                        else if (dtCol.ColumnName == TABLE.DailyControl.SecondaryEnhancementMediaId)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Secondary Enhancement Media";
                        }

                        //previous value
                        if (RowModified == false)
                        {
                            drAuditReport[ARTIFICIAL.OldValue] = "None";
                        }
                        else
                        {
                            if (dtCol.ColumnName == TABLE.DailyControl.CommercialTemplateIndicator)
                            {
                                if ((bool)dtDailyControlAudit.Rows[i + 1][dtCol] == true)
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "Commercial";
                                }
                                else
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "Non-Commercial";
                                }
                            }
                            else if (dtCol.ColumnName == TABLE.DailyControl.RackNamesIndicator)
                            {
                                if ((bool)dtDailyControlAudit.Rows[i + 1][dtCol] == true)
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "A-Z";
                                }
                                else
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "1-26";
                                }
                            }
                            else if (dtCol.ColumnName == TABLE.DailyControl.DailyAlertTime)
                            {
                                drAuditReport[ARTIFICIAL.OldValue] = ((DateTime)dtDailyControlAudit.Rows[i + 1][dtCol]).ToString("t");
                            }
                            else if (dtCol.ColumnName == TABLE.DailyControl.PrimaryEnhancementMediaId ||
                                     dtCol.ColumnName == TABLE.DailyControl.SecondaryEnhancementMediaId)
                            {
                                if (dtDailyControlAudit.Rows[i + 1].IsNull(dtCol) == false)
                                {
                                    DataTable dtEnhancementMedia = new BOL.EnhancementMedia().GetEnhancementMediaDataTable();
                                    if (dtEnhancementMedia.Select(TABLE.EnhancementMedia.EnhancementMediaId + " = '" + (int)dtDailyControlAudit.Rows[i + 1][dtCol] + "'").Length > 0)
                                    {
                                        drAuditReport[ARTIFICIAL.OldValue] = dtEnhancementMedia.Select(TABLE.EnhancementMedia.EnhancementMediaId + " = '" + (int)dtDailyControlAudit.Rows[i + 1][dtCol] + "'")[0][TABLE.EnhancementMedia.EnhancementMediaName].ToString();
                                    }
                                    else
                                        continue;
                                }
                                else
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "None";
                                }
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.OldValue] = dtDailyControlAudit.Rows[i + 1][dtCol].ToString().Trim();
                            }
                        }

                        //new value
                        if (dtCol.ColumnName == TABLE.DailyControl.CommercialTemplateIndicator)
                        {
                            if ((bool)dtDailyControlAudit.Rows[i][dtCol] == true)
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "Commercial";
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "Non-Commercial";
                            }
                        }
                        else if (dtCol.ColumnName == TABLE.DailyControl.RackNamesIndicator)
                        {
                            if ((bool)dtDailyControlAudit.Rows[i][dtCol] == true)
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "A-Z";
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "1-26";
                            }
                        }
                        else if (dtCol.ColumnName == TABLE.DailyControl.DailyAlertTime)
                        {
                            drAuditReport[ARTIFICIAL.NewValue] = ((DateTime)dtDailyControlAudit.Rows[i][dtCol]).ToString("t");
                        }
                        else if (dtCol.ColumnName == TABLE.DailyControl.PrimaryEnhancementMediaId ||
                            dtCol.ColumnName == TABLE.DailyControl.SecondaryEnhancementMediaId)
                        {
                            if (dtDailyControlAudit.Rows[i].IsNull(dtCol) == false)
                            {
                                DataTable dtEnhancementMedia = new BOL.EnhancementMedia().GetEnhancementMediaDataTable();
                                if (dtEnhancementMedia.Select(TABLE.EnhancementMedia.EnhancementMediaId + " = '" + (int)dtDailyControlAudit.Rows[i][dtCol] + "'").Length > 0)
                                {
                                    drAuditReport[ARTIFICIAL.NewValue] = dtEnhancementMedia.Select(TABLE.EnhancementMedia.EnhancementMediaId + " = '" + (int)dtDailyControlAudit.Rows[i][dtCol] + "'")[0][TABLE.EnhancementMedia.EnhancementMediaName].ToString();
                                }
                                else
                                    continue;
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "None";
                            }
                        }
                        else
                        {
                            drAuditReport[ARTIFICIAL.NewValue] = dtDailyControlAudit.Rows[i][dtCol].ToString().Trim();
                        }

                        try
                        {
                            BOL.VbecsUser LastUpdateUser = new BOL.VbecsUser(dtDailyControlAudit.Rows[i][TABLE.DailyControl.LastUpdateUser].ToString().Trim());
                            if (LastUpdateUser.UserName == null)
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = dtDailyControlAudit.Rows[i][TABLE.DailyControl.LastUpdateUser];
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = LastUpdateUser.UserName;
                            }
                        }
                        catch
                        {
                            drAuditReport[ARTIFICIAL.AuditUser] = dtDailyControlAudit.Rows[i][TABLE.DailyControl.LastUpdateUser];
                        }

                        drAuditReport[ARTIFICIAL.AuditDate] = dtDailyControlAudit.Rows[i][ARTIFICIAL.MirrorDate];

                        drAuditReport[ARTIFICIAL.Comment] = dtDailyControlAudit.Rows[i][TABLE.DailyControl.Comments].ToString().Trim();
                        this._dtAuditReport.Rows.Add(drAuditReport);
                    }
                }
            }

            DataTable dtReagentTestParameterAudit = DAL.AuditTrail.GetReagentTestParameterAudit(this.StartDate, this.EndDate);

            //generating one audit report row for each changed field
            nCount = dtReagentTestParameterAudit.Rows.Count;
            for (int i = 0; i < nCount; i++)
            {
                if ((DateTime)dtReagentTestParameterAudit.Rows[i][ARTIFICIAL.MirrorDate] < this.StartDate)
                    continue;
                if (i == dtReagentTestParameterAudit.Rows.Count - 1)
                {
                    RowModified = false;
                }
                else
                {
                    //if there was a change in configuration
                    //CR 3305 changes to minimum reactions will be captured properly
                    if (dtReagentTestParameterAudit.Rows[i][TABLE.ReagentTestParameter.ReagentTypeId].ToString() == dtReagentTestParameterAudit.Rows[i + 1][TABLE.ReagentTestParameter.ReagentTypeId].ToString() &&
                        (dtReagentTestParameterAudit.Rows[i][TABLE.ReagentTestParameter.SetVialId].ToString() == dtReagentTestParameterAudit.Rows[i + 1][TABLE.ReagentTestParameter.SetVialId].ToString() ||
                         (dtReagentTestParameterAudit.Rows[i].IsNull(TABLE.ReagentTestParameter.SetVialId) && dtReagentTestParameterAudit.Rows[i + 1].IsNull(TABLE.ReagentTestParameter.SetVialId))) &&
                        dtReagentTestParameterAudit.Rows[i][TABLE.ReagentTestParameter.TestWithName].ToString() == dtReagentTestParameterAudit.Rows[i + 1][TABLE.ReagentTestParameter.TestWithName].ToString() &&
                        (dtReagentTestParameterAudit.Rows[i][TABLE.ReagentTestParameter.DailyUseIndicator].ToString() != dtReagentTestParameterAudit.Rows[i + 1][TABLE.ReagentTestParameter.DailyUseIndicator].ToString() ||
                        dtReagentTestParameterAudit.Rows[i][TABLE.ReagentTestParameter.ExpectedMinimumReactions].ToString() != dtReagentTestParameterAudit.Rows[i + 1][TABLE.ReagentTestParameter.ExpectedMinimumReactions].ToString()))
                    {
                        RowModified = true;
                    }
                    else if (dtReagentTestParameterAudit.Rows[i][TABLE.ReagentTestParameter.ReagentTypeId].ToString() != dtReagentTestParameterAudit.Rows[i + 1][TABLE.ReagentTestParameter.ReagentTypeId].ToString() ||
                        dtReagentTestParameterAudit.Rows[i][TABLE.ReagentTestParameter.TestWithName].ToString() != dtReagentTestParameterAudit.Rows[i + 1][TABLE.ReagentTestParameter.TestWithName].ToString() ||
                        dtReagentTestParameterAudit.Rows[i][TABLE.ReagentTestParameter.SetVialId].ToString() != dtReagentTestParameterAudit.Rows[i + 1][TABLE.ReagentTestParameter.SetVialId].ToString())
                    {
                        RowModified = false;
                    }
                    else
                        continue;
                }

                foreach (DataColumn dtCol in dtReagentTestParameterAudit.Columns)
                {
                    if (dtCol.ColumnName == TABLE.ReagentTestParameter.DailyUseIndicator ||
                        dtCol.ColumnName == TABLE.ReagentTestParameter.ExpectedMinimumReactions)
                    {
                        //skip columns that have not changed the value
                        if (RowModified == true)
                        {
                            if (dtReagentTestParameterAudit.Rows[i][dtCol].Equals(dtReagentTestParameterAudit.Rows[i + 1][dtCol]))
                            {
                                continue;
                            }
                        }

                        DataRow drAuditReport = this._dtAuditReport.NewRow();
                        drAuditReport[ARTIFICIAL.AuditType] = AuditTrailConst.AuditTrailType.QCConfiguration;
                        drAuditReport[ARTIFICIAL.AuditIdentifier] = dtReagentTestParameterAudit.Rows[i][TABLE.ReagentType.ReagentTypeName].ToString().Trim();
                        if (dtReagentTestParameterAudit.Rows[i][TABLE.ReagentTestParameter.SetVialId].ToString().Trim() != "")
                        {
                            drAuditReport[ARTIFICIAL.AuditIdentifier] += " " + dtReagentTestParameterAudit.Rows[i][TABLE.ReagentTestParameter.SetVialId].ToString().Trim();
                        }
                        drAuditReport[ARTIFICIAL.AuditIdentifier] += " Tested With " + dtReagentTestParameterAudit.Rows[i][TABLE.ReagentTestParameter.TestWithName].ToString().Trim();

                        if (dtCol.ColumnName == TABLE.ReagentTestParameter.DailyUseIndicator)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Status Indicator";
                        }
                        else if (dtCol.ColumnName == TABLE.ReagentTestParameter.ExpectedMinimumReactions)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Minimum reaction entry";
                        }

                        //previous value
                        if (RowModified == false)
                        {
                            drAuditReport[ARTIFICIAL.OldValue] = "None";
                        }
                        else
                        {
                            if (dtCol.ColumnName == TABLE.ReagentTestParameter.DailyUseIndicator)
                            {
                                if ((bool)dtReagentTestParameterAudit.Rows[i + 1][dtCol] == true)
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "Required";
                                }
                                else
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "Not Required";
                                }
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.OldValue] = dtReagentTestParameterAudit.Rows[i + 1][dtCol].ToString().Trim();
                            }
                        }

                        //new value
                        if (dtCol.ColumnName == TABLE.ReagentTestParameter.DailyUseIndicator)
                        {
                            if ((bool)dtReagentTestParameterAudit.Rows[i][dtCol] == true)
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "Required";
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "Not Required";
                            }
                        }
                        else
                        {
                            drAuditReport[ARTIFICIAL.NewValue] = dtReagentTestParameterAudit.Rows[i][dtCol].ToString().Trim();
                        }

                        try
                        {
                            BOL.VbecsUser LastUpdateUser = new BOL.VbecsUser(dtReagentTestParameterAudit.Rows[i][TABLE.ReagentTestParameter.LastUpdateUser].ToString().Trim());
                            if (LastUpdateUser.UserName == null)
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = dtReagentTestParameterAudit.Rows[i][TABLE.ReagentTestParameter.LastUpdateUser];
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = LastUpdateUser.UserName;
                            }
                        }
                        catch
                        {
                            drAuditReport[ARTIFICIAL.AuditUser] = dtReagentTestParameterAudit.Rows[i][TABLE.ReagentTestParameter.LastUpdateUser];
                        }

                        drAuditReport[ARTIFICIAL.AuditDate] = dtReagentTestParameterAudit.Rows[i][ARTIFICIAL.MirrorDate];
                        drAuditReport[ARTIFICIAL.Comment] = "";

                        this._dtAuditReport.Rows.Add(drAuditReport);
                    }
                }
            }
        }

        /// <summary>
        /// Generates the DataTable used as a source for Division Audit Report
        /// </summary>
        private void GetConfigureTestingAuditReport()
        {
            bool RowModified = false;
            DataTable dtConfigureTestingAudit = DAL.AuditTrail.GetConfigureTestingAudit(this.StartDate, this.EndDate);
            int nCount = 0;
            //generating one audit report row for each changed field
            for (int i = 0; i < nCount; i++)
            {
                if ((DateTime)dtConfigureTestingAudit.Rows[i][ARTIFICIAL.MirrorDate] < this.StartDate)
                    continue;
                if (i == dtConfigureTestingAudit.Rows.Count - 1)
                {
                    RowModified = false;
                }
                else
                {
                    //do not compare records of two different blood units
                    //if the record has just been added and there is no history of changes
                    if (dtConfigureTestingAudit.Rows[i][TABLE.VamcDivision.DivisionCode].ToString() != dtConfigureTestingAudit.Rows[i + 1][TABLE.VamcDivision.DivisionCode].ToString())
                    {
                        RowModified = false;
                    }
                    else
                    {
                        RowModified = true;
                    }
                }

                foreach (DataColumn dtCol in dtConfigureTestingAudit.Columns)
                {
                    if (dtCol.ColumnName == TABLE.VamcDivision.AntibodyScreenDisplayCode)
                    {
                        //skip columns that have not changed the value
                        if (RowModified == true)
                        {
                            if (dtConfigureTestingAudit.Rows[i][dtCol].Equals(dtConfigureTestingAudit.Rows[i + 1][dtCol]))
                            {
                                continue;
                            }
                        }

                        DataRow drAuditReport = this._dtAuditReport.NewRow();
                        drAuditReport[ARTIFICIAL.AuditType] = AuditTrailConst.AuditTrailType.TestingConfiguration;
                        drAuditReport[ARTIFICIAL.AuditIdentifier] = dtConfigureTestingAudit.Rows[i][TABLE.VamcDivision.DivisionName].ToString().Trim();
                        drAuditReport[ARTIFICIAL.ChangedFieldName] = "Antibody Screen";

                        //previous value
                        if (RowModified == false)
                        {
                            drAuditReport[ARTIFICIAL.OldValue] = "None";
                        }
                        else
                        {
                            if (dtCol.ColumnName == TABLE.VamcDivision.AntibodyScreenDisplayCode)
                            {
                                drAuditReport[ARTIFICIAL.OldValue] = dtConfigureTestingAudit.Rows[i + 1][TABLE.AntibodyScreenDisplay.AntibodyScreenDisplayText].ToString().Trim();
                            }
                        }

                        //new value
                        if (dtCol.ColumnName == TABLE.VamcDivision.AntibodyScreenDisplayCode)
                        {
                            drAuditReport[ARTIFICIAL.NewValue] = dtConfigureTestingAudit.Rows[i][TABLE.AntibodyScreenDisplay.AntibodyScreenDisplayText].ToString().Trim();
                        }

                        try
                        {
                            BOL.VbecsUser LastUpdateUser = new BOL.VbecsUser(dtConfigureTestingAudit.Rows[i][TABLE.VamcDivision.LastUpdateUser].ToString().Trim());
                            if (LastUpdateUser.UserName == null)
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = dtConfigureTestingAudit.Rows[i][TABLE.VamcDivision.LastUpdateUser];
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = LastUpdateUser.UserName;
                            }
                        }
                        catch
                        {
                            drAuditReport[ARTIFICIAL.AuditUser] = dtConfigureTestingAudit.Rows[i][TABLE.VamcDivision.LastUpdateUser];
                        }

                        drAuditReport[ARTIFICIAL.AuditDate] = dtConfigureTestingAudit.Rows[i][ARTIFICIAL.MirrorDate];
                        drAuditReport[ARTIFICIAL.Comment] = "";

                        this._dtAuditReport.Rows.Add(drAuditReport);
                    }
                }
            }
        }

        /// <summary>
        /// Generates the DataTable used as a source for Maintaining Reagent Inventory Levels Audit Report
        /// </summary>
        private void GetReagentTypeParameterAuditReport()
        {
            bool RowModified = false;
            DataTable dtReagentTypeParameterAudit = DAL.AuditTrail.GetReagentTypeParameterAudit(this.StartDate, this.EndDate);
            int nCount = 0;
            //generating one audit report row for each changed field
            nCount = dtReagentTypeParameterAudit.Rows.Count;
            for (int i = 0; i < nCount; i++)
            {
                if ((DateTime)dtReagentTypeParameterAudit.Rows[i][ARTIFICIAL.MirrorDate] < this.StartDate)
                    continue;
                if (i == dtReagentTypeParameterAudit.Rows.Count - 1)
                {
                    RowModified = false;
                }
                else
                {
                    //do not compare records of two different blood units
                    //if the record has just been added and there is no history of changes
                    if (dtReagentTypeParameterAudit.Rows[i][TABLE.ReagentTypeParameter.ReagentTypeParameterGuid].ToString() != dtReagentTypeParameterAudit.Rows[i + 1][TABLE.ReagentTypeParameter.ReagentTypeParameterGuid].ToString())
                    {
                        RowModified = false;
                    }
                    else
                    {
                        RowModified = true;
                    }
                }

                foreach (DataColumn dtCol in dtReagentTypeParameterAudit.Columns)
                {
                    if (dtCol.ColumnName == TABLE.ReagentTypeParameter.MinimumStockLevel)
                    {
                        //skip columns that have not changed the value
                        if (RowModified == true)
                        {
                            if (dtReagentTypeParameterAudit.Rows[i][dtCol].Equals(dtReagentTypeParameterAudit.Rows[i + 1][dtCol]))
                            {
                                continue;
                            }
                        }

                        DataRow drAuditReport = this._dtAuditReport.NewRow();
                        drAuditReport[ARTIFICIAL.AuditType] = AuditTrailConst.AuditTrailType.ReagentInventoryLevel;
                        drAuditReport[ARTIFICIAL.AuditIdentifier] = dtReagentTypeParameterAudit.Rows[i][TABLE.ReagentType.ReagentTypeName].ToString().Trim();

                        drAuditReport[ARTIFICIAL.ChangedFieldName] = "Minimum inventory level";

                        //previous value
                        if (RowModified == false)
                        {
                            drAuditReport[ARTIFICIAL.OldValue] = "0";
                        }
                        else
                        {
                            drAuditReport[ARTIFICIAL.OldValue] = dtReagentTypeParameterAudit.Rows[i + 1][dtCol];
                        }

                        //new value
                        drAuditReport[ARTIFICIAL.NewValue] = dtReagentTypeParameterAudit.Rows[i][dtCol];

                        try
                        {
                            BOL.VbecsUser LastUpdateUser = new BOL.VbecsUser(dtReagentTypeParameterAudit.Rows[i][TABLE.ReagentTypeParameter.LastUpdateUser].ToString().Trim());
                            if (LastUpdateUser.UserName == null)
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = dtReagentTypeParameterAudit.Rows[i][TABLE.ReagentTypeParameter.LastUpdateUser];
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = LastUpdateUser.UserName;
                            }
                        }
                        catch
                        {
                            drAuditReport[ARTIFICIAL.AuditUser] = dtReagentTypeParameterAudit.Rows[i][TABLE.ReagentTypeParameter.LastUpdateUser];
                        }

                        drAuditReport[ARTIFICIAL.AuditDate] = dtReagentTypeParameterAudit.Rows[i][ARTIFICIAL.MirrorDate];
                        drAuditReport[ARTIFICIAL.Comment] = "";

                        this._dtAuditReport.Rows.Add(drAuditReport);
                    }
                }
            }
        }

        /// <summary>
        /// Generates the DataTable used as a source for User Alerts Audit Report
        /// </summary>
        private void GetUserAlertsAuditReport()
        {
            bool RowModified = false;
            DataTable dtOrderAlertTypes = BOL.DivisionAlerts.GetOrderAlertTypeList();
            DataTable dtUserAlertsAudit = DAL.AuditTrail.GetUserAlertsAudit(this.StartDate, this.EndDate);
            int nCount = 0;
            //generating one audit report row for each changed field
            nCount = dtUserAlertsAudit.Rows.Count;
            for (int i = 0; i < nCount; i++)
            {
                if ((DateTime)dtUserAlertsAudit.Rows[i][ARTIFICIAL.MirrorDate] < this.StartDate)
                    continue;
                if (i == dtUserAlertsAudit.Rows.Count - 1)
                {
                    RowModified = false;
                }
                else
                {
                    //do not compare records of two different blood units
                    //if the record has just been added and there is no history of changes
                    if (dtUserAlertsAudit.Rows[i][TABLE.VamcDivision.DivisionCode].ToString() != dtUserAlertsAudit.Rows[i + 1][TABLE.VamcDivision.DivisionCode].ToString())
                    {
                        RowModified = false;
                    }
                    else
                    {
                        RowModified = true;
                    }
                }

                foreach (DataColumn dtCol in dtUserAlertsAudit.Columns)
                {
                    if (dtCol.ColumnName == TABLE.VamcDivision.PrinterAlertsIndicator ||
                        dtCol.ColumnName == TABLE.VamcDivision.PrinterAlertsPrinterName ||
                        dtCol.ColumnName == TABLE.VamcDivision.PrinterOrderAlertTypeCode ||
                        dtCol.ColumnName == TABLE.VamcDivision.PrinterOrdersNeededHours ||
                        dtCol.ColumnName == TABLE.VamcDivision.IconAlertsIndicator ||
                        dtCol.ColumnName == TABLE.VamcDivision.IconRefreshRate ||
                        dtCol.ColumnName == TABLE.VamcDivision.IconOrderAlertTypeCode ||
                        dtCol.ColumnName == TABLE.VamcDivision.IconOrdersNeededHours)
                    {
                        //skip columns that have not changed the value
                        if (RowModified == true)
                        {
                            if (dtUserAlertsAudit.Rows[i][dtCol].Equals(dtUserAlertsAudit.Rows[i + 1][dtCol]) ||
                                (dtUserAlertsAudit.Rows[i].IsNull(dtCol) && dtUserAlertsAudit.Rows[i + 1].IsNull(dtCol)) ||
                                (dtUserAlertsAudit.Rows[i].IsNull(dtCol) && dtUserAlertsAudit.Rows[i + 1][dtCol].ToString().Trim() == "") ||
                                (dtUserAlertsAudit.Rows[i + 1].IsNull(dtCol) && dtUserAlertsAudit.Rows[i][dtCol].ToString().Trim() == "") ||
                                (dtUserAlertsAudit.Rows[i][dtCol].ToString().Trim() == "" && dtUserAlertsAudit.Rows[i + 1][dtCol].ToString().Trim() == ""))
                            {
                                continue;
                            }
                        }
                        else
                        {
                            if (dtUserAlertsAudit.Rows[i].IsNull(dtCol) ||
                                dtUserAlertsAudit.Rows[i][dtCol].ToString().Trim() == "")
                                continue;
                        }

                        DataRow drAuditReport = this._dtAuditReport.NewRow();
                        drAuditReport[ARTIFICIAL.AuditType] = AuditTrailConst.AuditTrailType.UserAlerts;
                        drAuditReport[ARTIFICIAL.AuditIdentifier] = dtUserAlertsAudit.Rows[i][TABLE.VamcDivision.DivisionName].ToString().Trim() + " (" + dtUserAlertsAudit.Rows[i][TABLE.VamcDivision.DivisionCode].ToString().Trim() + ")";

                        if (dtCol.ColumnName == TABLE.VamcDivision.PrinterAlertsIndicator)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Printer Alerts On";
                        }
                        else if (dtCol.ColumnName == TABLE.VamcDivision.PrinterAlertsPrinterName)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Printer Name";
                        }
                        else if (dtCol.ColumnName == TABLE.VamcDivision.PrinterOrderAlertTypeCode)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "(Printer) Order Alert Type";
                        }
                        else if (dtCol.ColumnName == TABLE.VamcDivision.PrinterOrdersNeededHours)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "(Printer) Orders Needed In";
                        }
                        else if (dtCol.ColumnName == TABLE.VamcDivision.IconAlertsIndicator)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Icon Alerts On";
                        }
                        else if (dtCol.ColumnName == TABLE.VamcDivision.IconRefreshRate)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Refresh Rate";
                        }
                        else if (dtCol.ColumnName == TABLE.VamcDivision.IconOrderAlertTypeCode)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "(Icon) Order Alert Type";
                        }
                        else if (dtCol.ColumnName == TABLE.VamcDivision.IconOrdersNeededHours)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "(Icon) Orders Needed In";
                        }

                        //previous value
                        if (RowModified == false)
                        {
                            drAuditReport[ARTIFICIAL.OldValue] = "None";
                        }
                        else
                        {
                            if (dtCol.ColumnName == TABLE.VamcDivision.PrinterAlertsIndicator ||
                                dtCol.ColumnName == TABLE.VamcDivision.IconAlertsIndicator)
                            {
                                if (dtUserAlertsAudit.Rows[i + 1].IsNull(dtCol) == false)
                                {
                                    if ((bool)dtUserAlertsAudit.Rows[i + 1][dtCol] == true)
                                    {
                                        drAuditReport[ARTIFICIAL.OldValue] = "On";
                                    }
                                    else
                                    {
                                        drAuditReport[ARTIFICIAL.OldValue] = "Off";
                                    }
                                }
                                else
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "Off";
                                }
                            }
                            else if (dtCol.ColumnName == TABLE.VamcDivision.PrinterOrderAlertTypeCode ||
                                     dtCol.ColumnName == TABLE.VamcDivision.IconOrderAlertTypeCode)
                            {
                                DataRow[] drAlertType = dtOrderAlertTypes.Select(TABLE.OrderAlertType.OrderAlertTypeCode + " = '" + dtUserAlertsAudit.Rows[i + 1][dtCol].ToString().Trim() + "'");
                                if (drAlertType.Length > 0)
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = drAlertType[0][TABLE.OrderAlertType.OrderAlertTypeText].ToString().Trim();
                                }
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.OldValue] = dtUserAlertsAudit.Rows[i + 1][dtCol];
                            }
                            if (drAuditReport[ARTIFICIAL.OldValue].ToString().Trim() == "")
                                drAuditReport[ARTIFICIAL.OldValue] = "None";
                        }

                        //new value
                        if (dtCol.ColumnName == TABLE.VamcDivision.PrinterAlertsIndicator ||
                            dtCol.ColumnName == TABLE.VamcDivision.IconAlertsIndicator)
                        {
                            if (dtUserAlertsAudit.Rows[i].IsNull(dtCol) == false)
                            {
                                if ((bool)dtUserAlertsAudit.Rows[i][dtCol] == true)
                                {
                                    drAuditReport[ARTIFICIAL.NewValue] = "On";
                                }
                                else
                                {
                                    drAuditReport[ARTIFICIAL.NewValue] = "Off";
                                }
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "Off";
                            }
                        }
                        else if (dtCol.ColumnName == TABLE.VamcDivision.PrinterOrderAlertTypeCode ||
                            dtCol.ColumnName == TABLE.VamcDivision.IconOrderAlertTypeCode)
                        {
                            DataRow[] drAlertType = dtOrderAlertTypes.Select(TABLE.OrderAlertType.OrderAlertTypeCode + " = '" + dtUserAlertsAudit.Rows[i][dtCol].ToString().Trim() + "'");
                            if (drAlertType.Length > 0)
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = drAlertType[0][TABLE.OrderAlertType.OrderAlertTypeText].ToString().Trim();
                            }
                        }
                        else
                        {
                            drAuditReport[ARTIFICIAL.NewValue] = dtUserAlertsAudit.Rows[i][dtCol];
                        }
                        if (drAuditReport[ARTIFICIAL.NewValue].ToString().Trim() == "")
                            drAuditReport[ARTIFICIAL.NewValue] = "None";

                        try
                        {
                            BOL.VbecsUser LastUpdateUser = new BOL.VbecsUser(dtUserAlertsAudit.Rows[i][TABLE.VamcDivision.LastUpdateUser].ToString().Trim());
                            if (LastUpdateUser.UserName == null)
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = dtUserAlertsAudit.Rows[i][TABLE.VamcDivision.LastUpdateUser];
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = LastUpdateUser.UserName;
                            }
                        }
                        catch
                        {
                            drAuditReport[ARTIFICIAL.AuditUser] = dtUserAlertsAudit.Rows[i][TABLE.VamcDivision.LastUpdateUser];
                        }

                        drAuditReport[ARTIFICIAL.AuditDate] = dtUserAlertsAudit.Rows[i][ARTIFICIAL.MirrorDate];
                        drAuditReport[ARTIFICIAL.Comment] = "";

                        this._dtAuditReport.Rows.Add(drAuditReport);
                    }
                }
            }
        }

        /// <summary>
        /// Generates the DataTable used as a source for BloodUnitStatus Audit Report
        /// </summary>
        private void GetBloodUnitStatusAuditReport()
        {
            bool RowModified = false;
            DataTable dtBloodUnitStatusAudit = DAL.AuditTrail.GetBloodUnitStatusAudit(this.StartDate, this.EndDate);
            int nCount = 0;
            //generating one audit report row for each changed field
            nCount = dtBloodUnitStatusAudit.Rows.Count;
            for (int i = 0; i < nCount; i++)
            {
                if ((DateTime)dtBloodUnitStatusAudit.Rows[i][ARTIFICIAL.MirrorDate] < this.StartDate)
                    continue;
                if (i == dtBloodUnitStatusAudit.Rows.Count - 1)
                {
                    continue;
                }
                else
                {
                    //do not compare records of two different blood units - just show the initial status of the unit
                    if (dtBloodUnitStatusAudit.Rows[i][TABLE.BloodUnitStatus.BloodUnitGuid].ToString() != dtBloodUnitStatusAudit.Rows[i + 1][TABLE.BloodUnitStatus.BloodUnitGuid].ToString() &&
                        (int)dtBloodUnitStatusAudit.Rows[i][ARTIFICIAL.MirrorId] != 0)
                    {
                        continue;
                    }
                    //if the record has just been added and there is no history of changes
                    if (dtBloodUnitStatusAudit.Rows[i][TABLE.BloodUnitStatus.BloodUnitGuid].ToString() != dtBloodUnitStatusAudit.Rows[i + 1][TABLE.BloodUnitStatus.BloodUnitGuid].ToString())
                    {
                        continue;
                    }
                    else
                    {
                        RowModified = true;
                    }
                }

                foreach (DataColumn dtCol in dtBloodUnitStatusAudit.Columns)
                {
                    if (dtCol.ColumnName == TABLE.BloodUnitStatus.BiohazardousWarningIndicator)
                    {
                        //skip columns that have not changed the value
                        if (RowModified == true)
                        {
                            if (dtBloodUnitStatusAudit.Rows[i][dtCol].Equals(dtBloodUnitStatusAudit.Rows[i + 1][dtCol]) ||
                                ((int)dtBloodUnitStatusAudit.Rows[i][TABLE.BloodUnitStatus.LastUpdateFunctionId] != (int)Common.UpdateFunction.UC013FrmInvalidateFinalStatus &&
                                 (int)dtBloodUnitStatusAudit.Rows[i][TABLE.BloodUnitStatus.LastUpdateFunctionId] != (int)Common.UpdateFunction.UC070FrmEditUnitProcessing))
                            {
                                continue;
                            }
                        }
                        else
                        {
                            if (dtCol.ColumnName == TABLE.BloodUnitStatus.BiohazardousWarningIndicator)
                                continue;
                        }

                        DataRow drAuditReport = this._dtAuditReport.NewRow();
                        drAuditReport[ARTIFICIAL.AuditType] = AuditTrailConst.AuditTrailType.BloodUnit;
                        drAuditReport[ARTIFICIAL.AuditIdentifier] = dtBloodUnitStatusAudit.Rows[i][TABLE.BloodUnit.EyeReadableUnitId].ToString().Trim() + ", " + dtBloodUnitStatusAudit.Rows[i][TABLE.BloodProduct.ProductCode].ToString().Trim() + ", " + dtBloodUnitStatusAudit.Rows[i][TABLE.BloodProduct.ProductName].ToString().Trim();

                        if (dtCol.ColumnName == TABLE.BloodUnitStatus.BiohazardousWarningIndicator)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Biohazardous Indicator";
                        }

                        if (RowModified == false)
                        {
                            drAuditReport[ARTIFICIAL.OldValue] = "None";
                        }
                        else
                        {
                            if (dtCol.ColumnName == TABLE.BloodUnitStatus.BiohazardousWarningIndicator)
                            {
                                if ((bool)dtBloodUnitStatusAudit.Rows[i + 1][dtCol] == true)
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "Yes";
                                }
                                else
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "No";
                                }
                            }
                        }

                        if (dtCol.ColumnName == TABLE.BloodUnitStatus.BiohazardousWarningIndicator)
                        {
                            if ((bool)dtBloodUnitStatusAudit.Rows[i][dtCol] == true)
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "Yes";
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "No";
                            }
                        }

                        try
                        {
                            BOL.VbecsUser LastUpdateUser = new BOL.VbecsUser(dtBloodUnitStatusAudit.Rows[i][TABLE.BloodUnitStatus.LastUpdateUser].ToString().Trim());
                            if (LastUpdateUser.UserName == null)
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = dtBloodUnitStatusAudit.Rows[i][TABLE.BloodUnitStatus.LastUpdateUser];
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = LastUpdateUser.UserName;
                            }
                        }
                        catch
                        {
                            drAuditReport[ARTIFICIAL.AuditUser] = dtBloodUnitStatusAudit.Rows[i][TABLE.BloodUnitStatus.LastUpdateUser];
                        }

                        drAuditReport[ARTIFICIAL.AuditDate] = dtBloodUnitStatusAudit.Rows[i][ARTIFICIAL.MirrorDate];
                        drAuditReport[ARTIFICIAL.Comment] = "";

                        //for transfused status add information about Patient
                        if (dtBloodUnitStatusAudit.Rows[i][TABLE.BloodUnitStatusCodeCurrent.UnitStatusCode].ToString().Trim() == Convert.ToString(Common.Utility.GetUnitStatusCodeCharFromEnum(Common.UnitStatusCode.Transfused)).Trim())
                        {
                            BOL.Patient transfusedPatient = BOL.PatientTransfusion.GetTransfusedPatient((Guid)dtBloodUnitStatusAudit.Rows[i][TABLE.BloodUnitStatus.BloodUnitGuid]);
                            drAuditReport[ARTIFICIAL.Comment] += "Transfused for patient: " + transfusedPatient.LastName + ", " + transfusedPatient.FirstName + "(" + transfusedPatient.DisplayPatientID + "). ";
                        }

                        //if the new comment has been entered diplay it
                        if (RowModified == true)
                        {
                            if (dtBloodUnitStatusAudit.Rows[i][TABLE.BloodUnitStatus.UnitStatusComments].ToString().Trim() !=
                                dtBloodUnitStatusAudit.Rows[i + 1][TABLE.BloodUnitStatus.UnitStatusComments].ToString().Trim())
                            {
                                drAuditReport[ARTIFICIAL.Comment] += dtBloodUnitStatusAudit.Rows[i][TABLE.BloodUnitStatus.UnitStatusComments].ToString().Trim();
                            }
                        }

                        this._dtAuditReport.Rows.Add(drAuditReport);
                    }
                }
            }
        }

        /// <summary>
        /// Generates the DataTable used as a source for BloodUnitModification Audit Report
        /// </summary>
        private void GetBloodUnitModificationAuditReport()
        {
            bool RowModified = false;
            DataTable dtBloodUnitModificationAudit = DAL.AuditTrail.GetBloodUnitModificationAudit(this.StartDate, this.EndDate);
            int nCount = 0;
            //generating one audit report row for each changed field
            nCount = dtBloodUnitModificationAudit.Rows.Count;
            for (int i = 0; i < nCount; i++)
            {
                if ((DateTime)dtBloodUnitModificationAudit.Rows[i][ARTIFICIAL.MirrorDate] < this.StartDate)
                    continue;
                if (i == dtBloodUnitModificationAudit.Rows.Count - 1)
                {
                    if ((bool)dtBloodUnitModificationAudit.Rows[i][TABLE.BloodUnitModification.OriginalUnitIndicator] == true)
                        continue;
                    else
                        RowModified = false;
                }
                else
                {
                    //do not compare records of two different pooled units
                    //if the record has just been added and there is no history of changes
                    if (dtBloodUnitModificationAudit.Rows[i][TABLE.BloodUnitModification.FromBloodUnitGuid].ToString() != dtBloodUnitModificationAudit.Rows[i + 1][TABLE.BloodUnitModification.FromBloodUnitGuid].ToString())
                    {
                        if ((bool)dtBloodUnitModificationAudit.Rows[i][TABLE.BloodUnitModification.OriginalUnitIndicator] == true)
                            continue;
                        else
                            RowModified = false;
                    }
                    else if (dtBloodUnitModificationAudit.Rows[i][TABLE.BloodUnitModification.FromBloodUnitGuid].ToString() == dtBloodUnitModificationAudit.Rows[i + 1][TABLE.BloodUnitModification.FromBloodUnitGuid].ToString() &&
                             dtBloodUnitModificationAudit.Rows[i][TABLE.BloodUnitModification.ToBloodUnitGuid].ToString() == dtBloodUnitModificationAudit.Rows[i + 1][TABLE.BloodUnitModification.ToBloodUnitGuid].ToString() &&
                             dtBloodUnitModificationAudit.Rows[i][TABLE.BloodUnitModification.RecordStatusCode].ToString().Trim() != dtBloodUnitModificationAudit.Rows[i + 1][TABLE.BloodUnitModification.RecordStatusCode].ToString().Trim())
                    {
                        RowModified = true;
                    }
                    else
                        continue;
                }


                DataRow drAuditReport = this._dtAuditReport.NewRow();
                drAuditReport[ARTIFICIAL.AuditType] = AuditTrailConst.AuditTrailType.BloodUnit;
                drAuditReport[ARTIFICIAL.AuditIdentifier] = dtBloodUnitModificationAudit.Rows[i][TABLE.BloodUnit.EyeReadableUnitId].ToString().Trim() + ", " + dtBloodUnitModificationAudit.Rows[i][TABLE.BloodProduct.ProductCode].ToString().Trim() + ", " + dtBloodUnitModificationAudit.Rows[i][TABLE.BloodProduct.ProductName].ToString().Trim();


                DataTable dtBloodUnit = DAL.BloodUnit.GetBloodUnitByGuid((Guid)dtBloodUnitModificationAudit.Rows[i][TABLE.BloodUnitModification.FromBloodUnitGuid]);
                if (dtBloodUnit.Rows.Count > 0)
                    drAuditReport[ARTIFICIAL.ChangedFieldName] = dtBloodUnit.Rows[0][TABLE.BloodUnit.EyeReadableUnitId].ToString().Trim() + ",   " + dtBloodUnit.Rows[0][TABLE.BloodUnit.UnitProductCode].ToString().Trim(); ;

                if (RowModified == false)
                {
                    drAuditReport[ARTIFICIAL.OldValue] = "None";
                    drAuditReport[ARTIFICIAL.NewValue] = "Added to the Pool";
                }
                else
                {
                    drAuditReport[ARTIFICIAL.OldValue] = "Pooled";
                    drAuditReport[ARTIFICIAL.NewValue] = "Removed from the Pool";
                }

                try
                {
                    BOL.VbecsUser LastUpdateUser = new BOL.VbecsUser(dtBloodUnitModificationAudit.Rows[i][TABLE.BloodUnitStatus.LastUpdateUser].ToString().Trim());
                    if (LastUpdateUser.UserName == null)
                    {
                        drAuditReport[ARTIFICIAL.AuditUser] = dtBloodUnitModificationAudit.Rows[i][TABLE.BloodUnitStatus.LastUpdateUser];
                    }
                    else
                    {
                        drAuditReport[ARTIFICIAL.AuditUser] = LastUpdateUser.UserName;
                    }
                }
                catch
                {
                    drAuditReport[ARTIFICIAL.AuditUser] = dtBloodUnitModificationAudit.Rows[i][TABLE.BloodUnitStatus.LastUpdateUser];
                }

                drAuditReport[ARTIFICIAL.AuditDate] = dtBloodUnitModificationAudit.Rows[i][ARTIFICIAL.MirrorDate];
                drAuditReport[ARTIFICIAL.Comment] = dtBloodUnitModificationAudit.Rows[i][TABLE.BloodUnitModification.Comments];

                this._dtAuditReport.Rows.Add(drAuditReport);
            }
        }

        /// <summary>
        /// Generates the DataTable used as a source for BloodUnit Audit Report
        /// </summary>
        private void GetBloodUnitAuditReport()
        {
            bool RowModified = false;
            DataTable dtBloodUnitAudit = DAL.AuditTrail.GetBloodUnitAudit(this.StartDate, this.EndDate);
            int nCount = 0;
            nCount = dtBloodUnitAudit.Rows.Count;
            //generating one audit report row for each changed field
            for (int i = 0; i < nCount; i++)
            {
                if ((DateTime)dtBloodUnitAudit.Rows[i][ARTIFICIAL.MirrorDate] < this.StartDate)
                    continue;
                if (i == dtBloodUnitAudit.Rows.Count - 1)
                {
                    continue;
                }
                else
                {
                    //do not compare records of two different blood units
                    if (dtBloodUnitAudit.Rows[i][TABLE.BloodUnit.BloodUnitGuid].ToString() != dtBloodUnitAudit.Rows[i + 1][TABLE.BloodUnit.BloodUnitGuid].ToString() &&
                        (int)dtBloodUnitAudit.Rows[i][ARTIFICIAL.MirrorId] != 0)
                    {
                        continue;
                    }
                    //if the record has just been added and there is no history of changes
                    else if (dtBloodUnitAudit.Rows[i][TABLE.BloodUnit.BloodUnitGuid].ToString() != dtBloodUnitAudit.Rows[i + 1][TABLE.BloodUnit.BloodUnitGuid].ToString() &&
                        (int)dtBloodUnitAudit.Rows[i][ARTIFICIAL.MirrorId] == 0)
                    {
                        continue;
                    }
                    else
                    {
                        RowModified = true;
                    }
                }

                foreach (DataColumn dtCol in dtBloodUnitAudit.Columns)
                {
                    if (dtCol.ColumnName == TABLE.BloodUnit.OriginalVolume ||
                        dtCol.ColumnName == TABLE.BloodUnit.DiscardedPlasmaVolume ||
                        dtCol.ColumnName == TABLE.BloodUnit.RecordStatusCode)
                    {
                        //skip columns that have not changed the value
                        if (RowModified == true)
                        {
                            if (dtBloodUnitAudit.Rows[i][dtCol].Equals(dtBloodUnitAudit.Rows[i + 1][dtCol]) ||
                                (dtBloodUnitAudit.Rows[i].IsNull(dtCol) == true &&
                                dtBloodUnitAudit.Rows[i + 1].IsNull(dtCol) == true) ||
                                (int)dtBloodUnitAudit.Rows[i][TABLE.BloodUnit.LastUpdateFunctionId] != (int)Common.UpdateFunction.UC070FrmEditUnitProcessing)
                            {
                                continue;
                            }

                            //do not show changes from Null to 0
                            if (dtCol.ColumnName == TABLE.BloodUnit.OriginalVolume ||
                                dtCol.ColumnName == TABLE.BloodUnit.DiscardedPlasmaVolume)
                            {
                                if (dtBloodUnitAudit.Rows[i].IsNull(dtCol) == false)
                                {
                                    if (dtBloodUnitAudit.Rows[i][dtCol].ToString().Trim() == "0" &&
                                        dtBloodUnitAudit.Rows[i + 1].IsNull(dtCol) == true)
                                    {
                                        continue;
                                    }
                                }

                            }
                        }
                        else
                        {
                            if (dtBloodUnitAudit.Rows[i].IsNull(dtCol) ||
                                dtCol.ColumnName == TABLE.BloodUnit.RecordStatusCode)
                                continue;
                        }

                        DataRow drAuditReport = this._dtAuditReport.NewRow();
                        drAuditReport[ARTIFICIAL.AuditType] = AuditTrailConst.AuditTrailType.BloodUnit;
                        drAuditReport[ARTIFICIAL.AuditIdentifier] = dtBloodUnitAudit.Rows[i][TABLE.BloodUnit.EyeReadableUnitId].ToString().Trim() + ", " + dtBloodUnitAudit.Rows[i][TABLE.BloodProduct.ProductCode].ToString().Trim() + ", " + dtBloodUnitAudit.Rows[i][TABLE.BloodProduct.ProductName].ToString().Trim();

                        if (dtCol.ColumnName == TABLE.BloodUnit.OriginalVolume)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Unit Volume";
                        }
                        else if (dtCol.ColumnName == TABLE.BloodUnit.DiscardedPlasmaVolume)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Discarded Plasma Volume";
                        }
                        else if (dtCol.ColumnName == TABLE.BloodUnit.RecordStatusCode)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Unit Record Inactivation Indicator";
                        }

                        if (RowModified == false)
                        {
                            drAuditReport[ARTIFICIAL.OldValue] = "None";
                        }
                        else
                        {
                            if (dtCol.ColumnName == TABLE.BloodUnit.RecordStatusCode)
                            {
                                if (dtBloodUnitAudit.Rows[i + 1][dtCol].ToString() == Convert.ToString(Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Inactive)))
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "Inactive";
                                }
                                else
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "Active";
                                }
                            }
                            else
                            {
                                if (dtBloodUnitAudit.Rows[i + 1].IsNull(dtCol) == false)
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = dtBloodUnitAudit.Rows[i + 1][dtCol];
                                }
                                else
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "None";
                                }
                            }
                        }

                        if (dtCol.ColumnName == TABLE.BloodUnit.RecordStatusCode)
                        {
                            if (dtBloodUnitAudit.Rows[i][dtCol].ToString() == Convert.ToString(Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Inactive)))
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "Inactive";
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "Active";
                            }
                        }
                        else
                        {
                            if (dtBloodUnitAudit.Rows[i].IsNull(dtCol) == false)
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = dtBloodUnitAudit.Rows[i][dtCol];
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "None";
                            }
                        }

                        try
                        {
                            BOL.VbecsUser LastUpdateUser = new BOL.VbecsUser(dtBloodUnitAudit.Rows[i][TABLE.BloodUnitStatus.LastUpdateUser].ToString().Trim());
                            if (LastUpdateUser.UserName == null)
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = dtBloodUnitAudit.Rows[i][TABLE.BloodUnitStatus.LastUpdateUser];
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = LastUpdateUser.UserName;
                            }
                        }
                        catch
                        {
                            drAuditReport[ARTIFICIAL.AuditUser] = dtBloodUnitAudit.Rows[i][TABLE.BloodUnitStatus.LastUpdateUser];
                        }

                        drAuditReport[ARTIFICIAL.AuditDate] = dtBloodUnitAudit.Rows[i][ARTIFICIAL.MirrorDate];
                        //CR 2134
                        if (dtCol.ColumnName == TABLE.BloodUnit.RecordStatusCode)
                        {
                            drAuditReport[ARTIFICIAL.Comment] = dtBloodUnitAudit.Rows[i][TABLE.BloodUnitMedia.Comments].ToString();
                        }
                        else
                        {
                            drAuditReport[ARTIFICIAL.Comment] = "";
                        }

                        this._dtAuditReport.Rows.Add(drAuditReport);
                    }
                }
            }
        }

        /// <summary>
        /// Generates the DataTable used as a source for BloodUnitMedia Audit Report
        /// </summary>
        private void GetBloodUnitMediaAuditReport()
        {
            bool RowModified = false;
            DataTable dtBloodUnitMediaAudit = DAL.AuditTrail.GetBloodUnitMediaAudit(this.StartDate, this.EndDate);
            int nCount = 0;
            //generating one audit report row for each changed field
            nCount = dtBloodUnitMediaAudit.Rows.Count;
            for (int i = 0; i < nCount; i++)
            {
                if ((DateTime)dtBloodUnitMediaAudit.Rows[i][ARTIFICIAL.MirrorDate] < this.StartDate)
                    continue;
                if (i == dtBloodUnitMediaAudit.Rows.Count - 1)
                {
                    continue;
                }
                else
                {
                    //do not compare records of two different blood units
                    if (dtBloodUnitMediaAudit.Rows[i][TABLE.BloodUnit.BloodUnitGuid].ToString() != dtBloodUnitMediaAudit.Rows[i + 1][TABLE.BloodUnit.BloodUnitGuid].ToString() &&
                        (int)dtBloodUnitMediaAudit.Rows[i][ARTIFICIAL.MirrorId] != 0)
                    {
                        continue;
                    }
                    //if the record has just been added and there is no history of changes
                    else if (dtBloodUnitMediaAudit.Rows[i][TABLE.BloodUnit.BloodUnitGuid].ToString() != dtBloodUnitMediaAudit.Rows[i + 1][TABLE.BloodUnit.BloodUnitGuid].ToString() &&
                        (int)dtBloodUnitMediaAudit.Rows[i][ARTIFICIAL.MirrorId] == 0)
                    {
                        continue;
                    }
                    else
                    {
                        RowModified = true;
                    }
                }

                foreach (DataColumn dtCol in dtBloodUnitMediaAudit.Columns)
                {
                    if (dtCol.ColumnName == TABLE.BloodUnitMedia.DirectedPatientGuid)
                    {
                        //skip columns that have not changed the value
                        if (RowModified == true)
                        {
                            if (dtBloodUnitMediaAudit.Rows[i][dtCol].Equals(dtBloodUnitMediaAudit.Rows[i + 1][dtCol]) ||
                                (int)dtBloodUnitMediaAudit.Rows[i][TABLE.BloodUnitMedia.LastUpdateFunctionId] != (int)Common.UpdateFunction.UC070FrmEditUnitProcessing)
                            {
                                continue;
                            }
                        }

                        DataRow drAuditReport = this._dtAuditReport.NewRow();
                        drAuditReport[ARTIFICIAL.AuditType] = AuditTrailConst.AuditTrailType.BloodUnit;
                        drAuditReport[ARTIFICIAL.AuditIdentifier] = dtBloodUnitMediaAudit.Rows[i][TABLE.BloodUnit.EyeReadableUnitId].ToString().Trim() + ", " + dtBloodUnitMediaAudit.Rows[i][TABLE.BloodProduct.ProductCode].ToString().Trim() + ", " + dtBloodUnitMediaAudit.Rows[i][TABLE.BloodProduct.ProductName].ToString().Trim();

                        if (dtCol.ColumnName == TABLE.BloodUnitMedia.DirectedPatientGuid)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Restricted to Patient Name and Patient ID";
                        }

                        if (RowModified == false)
                        {
                            drAuditReport[ARTIFICIAL.OldValue] = "None";
                        }
                        else
                        {
                            drAuditReport[ARTIFICIAL.OldValue] = dtBloodUnitMediaAudit.Rows[i + 1][TABLE.Patient.PatientLastName].ToString() + ", " + dtBloodUnitMediaAudit.Rows[i + 1][TABLE.Patient.PatientFirstName].ToString() + " (" + dtBloodUnitMediaAudit.Rows[i + 1][TABLE.Patient.DisplayVistaPatientId].ToString() + ")";
                        }

                        drAuditReport[ARTIFICIAL.NewValue] = dtBloodUnitMediaAudit.Rows[i][TABLE.Patient.PatientLastName].ToString() + ", " + dtBloodUnitMediaAudit.Rows[i][TABLE.Patient.PatientFirstName].ToString() + " (" + dtBloodUnitMediaAudit.Rows[i][TABLE.Patient.DisplayVistaPatientId].ToString() + ")";

                        try
                        {
                            BOL.VbecsUser LastUpdateUser = new BOL.VbecsUser(dtBloodUnitMediaAudit.Rows[i][TABLE.BloodUnitStatus.LastUpdateUser].ToString().Trim());
                            if (LastUpdateUser.UserName == null)
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = dtBloodUnitMediaAudit.Rows[i][TABLE.BloodUnitStatus.LastUpdateUser];
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = LastUpdateUser.UserName;
                            }
                        }
                        catch
                        {
                            drAuditReport[ARTIFICIAL.AuditUser] = dtBloodUnitMediaAudit.Rows[i][TABLE.BloodUnitStatus.LastUpdateUser];
                        }

                        drAuditReport[ARTIFICIAL.AuditDate] = dtBloodUnitMediaAudit.Rows[i][ARTIFICIAL.MirrorDate];
                        drAuditReport[ARTIFICIAL.Comment] = "";

                        this._dtAuditReport.Rows.Add(drAuditReport);
                    }
                }
            }
        }

        /// <summary>
        /// Generates the DataTable used as a source for BloodUnitAntigen Audit Report
        /// </summary>
        private void GetBloodUnitAntigenAuditReport()
        {
            bool RowModified = false;
            DataTable dtBloodUnitAntigenAudit = DAL.AuditTrail.GetBloodUnitAntigenAudit(this.StartDate, this.EndDate);
            int nCount = 0;
            //generating one audit report row for each changed field
            nCount = dtBloodUnitAntigenAudit.Rows.Count;
            for (int i = 0; i < nCount; i++)
            {
                if ((DateTime)dtBloodUnitAntigenAudit.Rows[i][ARTIFICIAL.MirrorDate] < this.StartDate)
                    continue;
                if (i == dtBloodUnitAntigenAudit.Rows.Count - 1)
                {
                    RowModified = false;
                }
                else
                {
                    //CR 2614 Modified logic checking which records were changed and should be displayed on the report
                    //do not compare records of two different blood units
                    if (dtBloodUnitAntigenAudit.Rows[i][TABLE.BloodUnit.BloodUnitGuid].ToString() != dtBloodUnitAntigenAudit.Rows[i + 1][TABLE.BloodUnit.BloodUnitGuid].ToString())
                    {
                        RowModified = false;
                    }
                    else if (dtBloodUnitAntigenAudit.Rows[i][TABLE.AntigenType.AntigenTypeName].ToString() != dtBloodUnitAntigenAudit.Rows[i + 1][TABLE.AntigenType.AntigenTypeName].ToString())
                    {
                        RowModified = false;
                    }
                    else
                    {
                        RowModified = true;
                    }

                    //CR 2614 Skip auditing of UC_01 entries when RowModified == false
                    if ((int)dtBloodUnitAntigenAudit.Rows[i][TABLE.BloodUnitTest.LastUpdateFunctionId] != (int)Common.UpdateFunction.UC070FrmEditUnitProcessing && !RowModified)
                    {
                        continue;
                    }

                }


                //skip columns that have not changed the value
                if (RowModified == true)
                {
                    if ((dtBloodUnitAntigenAudit.Rows[i][TABLE.BloodUnitAntigen.RecordStatusCode].Equals(dtBloodUnitAntigenAudit.Rows[i + 1][TABLE.BloodUnitAntigen.RecordStatusCode]) &&
                        dtBloodUnitAntigenAudit.Rows[i][TABLE.BloodUnitAntigen.Positive].Equals(dtBloodUnitAntigenAudit.Rows[i + 1][TABLE.BloodUnitAntigen.Positive])) ||
                        (int)dtBloodUnitAntigenAudit.Rows[i][TABLE.BloodUnitTest.LastUpdateFunctionId] != (int)Common.UpdateFunction.UC070FrmEditUnitProcessing)
                    {
                        continue;
                    }
                }

                DataRow drAuditReport = this._dtAuditReport.NewRow();
                drAuditReport[ARTIFICIAL.AuditType] = AuditTrailConst.AuditTrailType.BloodUnit;
                drAuditReport[ARTIFICIAL.AuditIdentifier] = dtBloodUnitAntigenAudit.Rows[i][TABLE.BloodUnit.EyeReadableUnitId].ToString().Trim() + ", " + dtBloodUnitAntigenAudit.Rows[i][TABLE.BloodProduct.ProductCode].ToString().Trim() + ", " + dtBloodUnitAntigenAudit.Rows[i][TABLE.BloodProduct.ProductName].ToString().Trim();

                drAuditReport[ARTIFICIAL.ChangedFieldName] = "Unit Antigen Information (" + dtBloodUnitAntigenAudit.Rows[i][TABLE.AntigenType.AntigenTypeName].ToString().Trim() + ")";

                if (RowModified == false)
                {
                    drAuditReport[ARTIFICIAL.OldValue] = "None";
                }
                else
                {
                    if (dtBloodUnitAntigenAudit.Rows[i + 1][TABLE.BloodUnitAntigen.RecordStatusCode].ToString() == Convert.ToString(Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Inactive)))
                    {
                        drAuditReport[ARTIFICIAL.OldValue] = "None";
                    }
                    else
                    {
                        if ((bool)dtBloodUnitAntigenAudit.Rows[i + 1][TABLE.BloodUnitAntigen.Positive] == true)
                        {
                            drAuditReport[ARTIFICIAL.OldValue] = TABLE.BloodUnitAntigen.Positive;
                        }
                        else
                        {
                            drAuditReport[ARTIFICIAL.OldValue] = "Negative";
                        }
                    }
                }

                if (dtBloodUnitAntigenAudit.Rows[i][TABLE.BloodUnitAntigen.RecordStatusCode].ToString() == Convert.ToString(Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Inactive)))
                {
                    drAuditReport[ARTIFICIAL.NewValue] = "None";
                }
                else
                {
                    if ((bool)dtBloodUnitAntigenAudit.Rows[i][TABLE.BloodUnitAntigen.Positive] == true)
                    {
                        drAuditReport[ARTIFICIAL.NewValue] = TABLE.BloodUnitAntigen.Positive;
                    }
                    else
                    {
                        drAuditReport[ARTIFICIAL.NewValue] = "Negative";
                    }
                }

                try
                {
                    BOL.VbecsUser LastUpdateUser = new BOL.VbecsUser(dtBloodUnitAntigenAudit.Rows[i][TABLE.BloodUnitStatus.LastUpdateUser].ToString().Trim());
                    if (LastUpdateUser.UserName == null)
                    {
                        drAuditReport[ARTIFICIAL.AuditUser] = dtBloodUnitAntigenAudit.Rows[i][TABLE.BloodUnitStatus.LastUpdateUser];
                    }
                    else
                    {
                        drAuditReport[ARTIFICIAL.AuditUser] = LastUpdateUser.UserName;
                    }
                }
                catch
                {
                    drAuditReport[ARTIFICIAL.AuditUser] = dtBloodUnitAntigenAudit.Rows[i][TABLE.BloodUnitStatus.LastUpdateUser];
                }

                drAuditReport[ARTIFICIAL.AuditDate] = dtBloodUnitAntigenAudit.Rows[i][ARTIFICIAL.MirrorDate];
                drAuditReport[ARTIFICIAL.Comment] = "";
                this._dtAuditReport.Rows.Add(drAuditReport);
            }
        }

        /// <summary>
        /// Generates the DataTable used as a source for BloodUnitTest Audit Report
        /// </summary>
        private void GetBloodUnitTestAuditReport()
        {
            bool RowModified = false;
            DataTable dtBloodUnitTestAudit = DAL.AuditTrail.GetBloodUnitTestAudit(this.StartDate, this.EndDate);
            int nCount = 0;
            //generating one audit report row for each changed field
            nCount = dtBloodUnitTestAudit.Rows.Count;
            for (int i = 0; i < nCount; i++)
            {
                if ((DateTime)dtBloodUnitTestAudit.Rows[i][ARTIFICIAL.MirrorDate] < this.StartDate)
                    continue;
                //skip rh interp - we will get it along with ABO interp
                if ((int)dtBloodUnitTestAudit.Rows[i][TABLE.BloodUnitTest.BloodTestTypeId] == (int)Common.TestType.RhInterp)
                    continue;

                if (i == dtBloodUnitTestAudit.Rows.Count - 1)
                {
                    continue;
                }
                else
                {
                    //do not compare records of two different blood units
                    if ((dtBloodUnitTestAudit.Rows[i][TABLE.BloodUnit.BloodUnitGuid].ToString() != dtBloodUnitTestAudit.Rows[i + 1][TABLE.BloodUnit.BloodUnitGuid].ToString() ||
                        (int)dtBloodUnitTestAudit.Rows[i][TABLE.BloodUnitTest.BloodTestTypeId] != (int)dtBloodUnitTestAudit.Rows[i + 1][TABLE.BloodUnitTest.BloodTestTypeId]) &&
                        (int)dtBloodUnitTestAudit.Rows[i][ARTIFICIAL.MirrorId] != 0)
                    {
                        continue;
                    }
                    //if the record has just been added and there is no history of changes
                    else if ((dtBloodUnitTestAudit.Rows[i][TABLE.BloodUnit.BloodUnitGuid].ToString() != dtBloodUnitTestAudit.Rows[i + 1][TABLE.BloodUnit.BloodUnitGuid].ToString() ||
                            (int)dtBloodUnitTestAudit.Rows[i][TABLE.BloodUnitTest.BloodTestTypeId] != (int)dtBloodUnitTestAudit.Rows[i + 1][TABLE.BloodUnitTest.BloodTestTypeId]) &&
                            (int)dtBloodUnitTestAudit.Rows[i][ARTIFICIAL.MirrorId] == 0)
                    {
                        continue;
                    }
                    else
                    {
                        RowModified = true;
                    }
                }


                //skip columns that have not changed the value
                if (RowModified == true)
                {
                    if ((dtBloodUnitTestAudit.Rows[i][TABLE.BloodUnitTest.RecordStatusCode].Equals(dtBloodUnitTestAudit.Rows[i + 1][TABLE.BloodUnitTest.RecordStatusCode]) &&
                        dtBloodUnitTestAudit.Rows[i][TABLE.BloodUnitTest.TestResultId].Equals(dtBloodUnitTestAudit.Rows[i + 1][TABLE.BloodUnitTest.TestResultId])) ||
                        (dtBloodUnitTestAudit.Rows[i].IsNull(TABLE.BloodUnitAntigen.RecordStatusCode) == true &&
                        dtBloodUnitTestAudit.Rows[i + 1].IsNull(TABLE.BloodUnitAntigen.RecordStatusCode) == true) ||
                        (int)dtBloodUnitTestAudit.Rows[i][TABLE.BloodUnitTest.LastUpdateFunctionId] != (int)Common.UpdateFunction.UC070FrmEditUnitProcessing)
                    {
                        continue;
                    }
                }
                else
                {
                    if (dtBloodUnitTestAudit.Rows[i].IsNull(TABLE.BloodUnitTest.RecordStatusCode) == true)
                        continue;
                }

                DataRow drAuditReport = this._dtAuditReport.NewRow();
                drAuditReport[ARTIFICIAL.AuditType] = AuditTrailConst.AuditTrailType.BloodUnit;
                drAuditReport[ARTIFICIAL.AuditIdentifier] = dtBloodUnitTestAudit.Rows[i][TABLE.BloodUnit.EyeReadableUnitId].ToString().Trim() + ", " + dtBloodUnitTestAudit.Rows[i][TABLE.BloodProduct.ProductCode].ToString().Trim() + ", " + dtBloodUnitTestAudit.Rows[i][TABLE.BloodProduct.ProductName].ToString().Trim();

                if ((int)dtBloodUnitTestAudit.Rows[i][TABLE.BloodUnitTest.BloodTestTypeId] == (int)Common.TestType.ABOInterp)
                {
                    drAuditReport[ARTIFICIAL.ChangedFieldName] = "Unit ABO/Rh Confirmation Test/Invalidation Indicator";
                }
                else if ((int)dtBloodUnitTestAudit.Rows[i][TABLE.BloodUnitTest.BloodTestTypeId] == (int)Common.TestType.CMV)
                {
                    drAuditReport[ARTIFICIAL.ChangedFieldName] = "Special Testing Information (CMV Negative)";
                }
                else if ((int)dtBloodUnitTestAudit.Rows[i][TABLE.BloodUnitTest.BloodTestTypeId] == (int)Common.TestType.SickleCell)
                {
                    drAuditReport[ARTIFICIAL.ChangedFieldName] = "Special Testing Information (Sickle Cell Negative)";
                }
                else
                {
                    drAuditReport[ARTIFICIAL.ChangedFieldName] = "Unit Antigen Typing Test/Invalidation Indicator (" + dtBloodUnitTestAudit.Rows[i][TABLE.BloodTestType.BloodTestName].ToString().Replace("Repeat", "").Replace(ARTIFICIAL.Interp, "").Trim() + ")";
                }

                if (RowModified == false)
                {
                    drAuditReport[ARTIFICIAL.OldValue] = "None";
                }
                else
                {
                    if (dtBloodUnitTestAudit.Rows[i + 1][TABLE.BloodUnitAntigen.RecordStatusCode].ToString() == Convert.ToString(Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.ErrorCorrected)))
                    {
                        drAuditReport[ARTIFICIAL.OldValue] = "Test Result Invalid";
                    }
                    else
                    {
                        if ((int)dtBloodUnitTestAudit.Rows[i + 1][TABLE.BloodUnitTest.BloodTestTypeId] == (int)Common.TestType.ABOInterp)
                        {
                            string AboRh = dtBloodUnitTestAudit.Rows[i + 1][TABLE.BloodUnitTest.TestResultId].ToString().Trim();
                            DataRow[] drBloodUnitTestRh = dtBloodUnitTestAudit.Select(TABLE.BloodUnitTest.ConfirmationWorklistUnitGuid + " = '" + dtBloodUnitTestAudit.Rows[i + 1][TABLE.BloodUnitTest.ConfirmationWorklistUnitGuid].ToString() + "' AND " + TABLE.BloodUnitTest.BloodTestTypeId + " = " + (int)Common.TestType.RhInterp);
                            if (drBloodUnitTestRh.Length > 0)
                            {
                                if (drBloodUnitTestRh[0][TABLE.BloodUnitTest.TestResultId].ToString().Trim() == "P")
                                {
                                    AboRh += " Pos";
                                }
                                else if (drBloodUnitTestRh[0][TABLE.BloodUnitTest.TestResultId].ToString().Trim() == "N")
                                {
                                    AboRh += " Neg";
                                }
                                else if (drBloodUnitTestRh[0][TABLE.BloodUnitTest.TestResultId].ToString().Trim() == "I")
                                {
                                    AboRh += " Inc";
                                }
                            }
                            drAuditReport[ARTIFICIAL.OldValue] = AboRh;
                        }
                        else if ((int)dtBloodUnitTestAudit.Rows[i + 1][TABLE.BloodUnitTest.BloodTestTypeId] == (int)Common.TestType.CMV ||
                                    (int)dtBloodUnitTestAudit.Rows[i + 1][TABLE.BloodUnitTest.BloodTestTypeId] == (int)Common.TestType.SickleCell)
                        {
                            if (dtBloodUnitTestAudit.Rows[i + 1].IsNull(TABLE.BloodUnitTest.TestResultId) == true ||
                                dtBloodUnitTestAudit.Rows[i + 1][TABLE.BloodUnitTest.TestResultId].ToString().Trim() == "P")
                                drAuditReport[ARTIFICIAL.OldValue] = "None";
                            else
                            {
                                if (dtBloodUnitTestAudit.Rows[i + 1][TABLE.BloodUnitTest.TestResultId].ToString().Trim() == "N")
                                    drAuditReport[ARTIFICIAL.OldValue] = "Negative";
                            }
                        }
                        else
                        {
                            if (dtBloodUnitTestAudit.Rows[i + 1][TABLE.BloodUnitTest.TestResultId].ToString().Trim() == "N")
                                drAuditReport[ARTIFICIAL.OldValue] = "Neg";
                            else if (dtBloodUnitTestAudit.Rows[i + 1][TABLE.BloodUnitTest.TestResultId].ToString().Trim() == "P")
                                drAuditReport[ARTIFICIAL.OldValue] = "Pos";
                            else if (dtBloodUnitTestAudit.Rows[i + 1][TABLE.BloodUnitTest.TestResultId].ToString().Trim() == "I")
                                drAuditReport[ARTIFICIAL.OldValue] = "Inc";
                        }
                    }
                }

                if (dtBloodUnitTestAudit.Rows[i][TABLE.BloodUnitAntigen.RecordStatusCode].ToString() == Convert.ToString(Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.ErrorCorrected)))
                {
                    drAuditReport[ARTIFICIAL.NewValue] = "Test Result Invalid";
                }
                else
                {
                    if ((int)dtBloodUnitTestAudit.Rows[i][TABLE.BloodUnitTest.BloodTestTypeId] == (int)Common.TestType.ABOInterp)
                    {
                        string AboRh = dtBloodUnitTestAudit.Rows[i][TABLE.BloodUnitTest.TestResultId].ToString().Trim();
                        DataRow[] drBloodUnitTestRh = dtBloodUnitTestAudit.Select(TABLE.BloodUnitTest.ConfirmationWorklistUnitGuid + " = '" + dtBloodUnitTestAudit.Rows[i][TABLE.BloodUnitTest.ConfirmationWorklistUnitGuid].ToString() + "' AND " + TABLE.BloodUnitTest.BloodTestTypeId + " = " + (int)Common.TestType.RhInterp);
                        if (drBloodUnitTestRh.Length > 0)
                        {
                            if (drBloodUnitTestRh[0][TABLE.BloodUnitTest.TestResultId].ToString().Trim() == "P")
                            {
                                AboRh += " Pos";
                            }
                            else if (drBloodUnitTestRh[0][TABLE.BloodUnitTest.TestResultId].ToString().Trim() == "N")
                            {
                                AboRh += " Neg";
                            }
                            else if (drBloodUnitTestRh[0][TABLE.BloodUnitTest.TestResultId].ToString().Trim() == "I")
                            {
                                AboRh += " Inc";
                            }
                        }
                        drAuditReport[ARTIFICIAL.NewValue] = AboRh;
                    }
                    else if ((int)dtBloodUnitTestAudit.Rows[i][TABLE.BloodUnitTest.BloodTestTypeId] == (int)Common.TestType.CMV ||
                                (int)dtBloodUnitTestAudit.Rows[i][TABLE.BloodUnitTest.BloodTestTypeId] == (int)Common.TestType.SickleCell)
                    {
                        if (dtBloodUnitTestAudit.Rows[i].IsNull(TABLE.BloodUnitTest.TestResultId) == true ||
                            dtBloodUnitTestAudit.Rows[i][TABLE.BloodUnitTest.TestResultId].ToString().Trim() == "P")
                            drAuditReport[ARTIFICIAL.NewValue] = "None";
                        else
                        {
                            if (dtBloodUnitTestAudit.Rows[i][TABLE.BloodUnitTest.TestResultId].ToString().Trim() == "N")
                                drAuditReport[ARTIFICIAL.NewValue] = "Negative";
                        }
                    }
                    else
                    {
                        if (dtBloodUnitTestAudit.Rows[i][TABLE.BloodUnitTest.TestResultId].ToString().Trim() == "N")
                            drAuditReport[ARTIFICIAL.NewValue] = "Neg";
                        else if (dtBloodUnitTestAudit.Rows[i][TABLE.BloodUnitTest.TestResultId].ToString().Trim() == "P")
                            drAuditReport[ARTIFICIAL.NewValue] = "Pos";
                        else if (dtBloodUnitTestAudit.Rows[i][TABLE.BloodUnitTest.TestResultId].ToString().Trim() == "I")
                            drAuditReport[ARTIFICIAL.NewValue] = "Inc";
                    }
                }

                try
                {
                    BOL.VbecsUser LastUpdateUser = new BOL.VbecsUser(dtBloodUnitTestAudit.Rows[i][TABLE.BloodUnitStatus.LastUpdateUser].ToString().Trim());
                    if (LastUpdateUser.UserName == null)
                    {
                        drAuditReport[ARTIFICIAL.AuditUser] = dtBloodUnitTestAudit.Rows[i][TABLE.BloodUnitStatus.LastUpdateUser];
                    }
                    else
                    {
                        drAuditReport[ARTIFICIAL.AuditUser] = LastUpdateUser.UserName;
                    }
                }
                catch
                {
                    drAuditReport[ARTIFICIAL.AuditUser] = dtBloodUnitTestAudit.Rows[i][TABLE.BloodUnitStatus.LastUpdateUser];
                }

                drAuditReport[ARTIFICIAL.AuditDate] = dtBloodUnitTestAudit.Rows[i][ARTIFICIAL.MirrorDate];
                drAuditReport[ARTIFICIAL.Comment] = dtBloodUnitTestAudit.Rows[i][TABLE.BloodUnitTest.TestComments].ToString().Trim();
                this._dtAuditReport.Rows.Add(drAuditReport);
            }
        }

        /// <summary>
        /// Generates the DataTable used as a source for UserRoles Audit Report
        /// </summary>
        private void GetUserRoleAuditReport()
        {
            bool RowModified = false;
            DataTable dtUserRoleAudit = DAL.AuditTrail.GetUserRoleAudit(this.StartDate, this.EndDate);
            int nCount = 0;
            //generating one audit report row for each changed field
            nCount = dtUserRoleAudit.Rows.Count;
            for (int i = 0; i < nCount; i++)
            {
                if ((DateTime)dtUserRoleAudit.Rows[i][ARTIFICIAL.MirrorDate] < this.StartDate)
                    continue;
                if (i == dtUserRoleAudit.Rows.Count - 1)
                {
                    RowModified = false;
                }
                else
                {
                    //do not compare records of two different blood units
                    //if the record has just been added and there is no history of changes
                    if (dtUserRoleAudit.Rows[i][TABLE.VbecsUser.UserName].ToString() != dtUserRoleAudit.Rows[i + 1][TABLE.VbecsUser.UserName].ToString())
                    {
                        RowModified = false;
                    }
                    else
                    {
                        RowModified = true;
                    }
                }

                foreach (DataColumn dtCol in dtUserRoleAudit.Columns)
                {
                    if (dtCol.ColumnName == TABLE.VbecsUserDivisionRole.UserRoleId ||
                        dtCol.ColumnName == TABLE.VbecsUserDivisionRole.RecordStatusCode)
                    {
                        //skip columns that have not changed the value
                        if (RowModified == true)
                        {
                            if (dtUserRoleAudit.Rows[i][dtCol].Equals(dtUserRoleAudit.Rows[i + 1][dtCol]))
                            {
                                continue;
                            }
                        }

                        DataRow drAuditReport = this._dtAuditReport.NewRow();
                        drAuditReport[ARTIFICIAL.AuditType] = AuditTrailConst.AuditTrailType.MaintainUser;
                        drAuditReport[ARTIFICIAL.AuditIdentifier] = dtUserRoleAudit.Rows[i][TABLE.VbecsUser.UserName].ToString().Trim() + " (" + dtUserRoleAudit.Rows[i][TABLE.VbecsUser.VbecsUserId].ToString().Trim() + ")";

                        if (dtCol.ColumnName == TABLE.VbecsUserDivisionRole.UserRoleId)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "User Role Security Setting";
                        }
                        else if (dtCol.ColumnName == TABLE.VbecsUserDivisionRole.RecordStatusCode)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Active Division Indicator";
                        }

                        if (RowModified == false)
                        {
                            drAuditReport[ARTIFICIAL.OldValue] = "None";
                        }
                        else
                        {
                            if (dtCol.ColumnName == TABLE.VbecsUserDivisionRole.UserRoleId)
                            {
                                drAuditReport[ARTIFICIAL.OldValue] = dtUserRoleAudit.Rows[i + 1][TABLE.UserRole.UserRoleName].ToString().Trim();
                            }
                            else if (dtCol.ColumnName == TABLE.VbecsUserDivisionRole.RecordStatusCode)
                            {
                                if (dtUserRoleAudit.Rows[i + 1][dtCol].ToString().Trim() == Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active).ToString().Trim())
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "Active";
                                }
                                else
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "Inactive";
                                }
                            }
                        }

                        if (dtCol.ColumnName == TABLE.VbecsUserDivisionRole.UserRoleId)
                        {
                            drAuditReport[ARTIFICIAL.NewValue] = dtUserRoleAudit.Rows[i][TABLE.UserRole.UserRoleName].ToString().Trim();
                        }
                        else if (dtCol.ColumnName == TABLE.VbecsUserDivisionRole.RecordStatusCode)
                        {
                            if (dtUserRoleAudit.Rows[i][dtCol].ToString().Trim() == Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active).ToString().Trim())
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "Active";
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "Inactive";
                            }
                        }

                        try
                        {
                            BOL.VbecsUser LastUpdateUser = new BOL.VbecsUser(dtUserRoleAudit.Rows[i][TABLE.BloodUnitStatus.LastUpdateUser].ToString().Trim());
                            if (LastUpdateUser.UserName == null)
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = dtUserRoleAudit.Rows[i][TABLE.BloodUnitStatus.LastUpdateUser];
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = LastUpdateUser.UserName;
                            }
                        }
                        catch
                        {
                            drAuditReport[ARTIFICIAL.AuditUser] = dtUserRoleAudit.Rows[i][TABLE.BloodUnitStatus.LastUpdateUser];
                        }

                        drAuditReport[ARTIFICIAL.AuditDate] = dtUserRoleAudit.Rows[i][ARTIFICIAL.MirrorDate];

                        drAuditReport[ARTIFICIAL.Comment] = "";

                        this._dtAuditReport.Rows.Add(drAuditReport);
                    }
                }
            }

            DataTable dtUserAudit = DAL.AuditTrail.GetUserAudit(this.StartDate, this.EndDate);

            //generating one audit report row for each changed field
            nCount = dtUserAudit.Rows.Count;
            for (int i = 0; i < nCount; i++)
            {
                if ((DateTime)dtUserAudit.Rows[i][ARTIFICIAL.MirrorDate] < this.StartDate)
                    continue;
                if (i == dtUserAudit.Rows.Count - 1)
                {
                    RowModified = false;
                }
                else
                {
                    //do not compare records of two different blood units
                    //if the record has just been added and there is no history of changes
                    if (dtUserAudit.Rows[i][TABLE.VbecsUser.UserName].ToString() != dtUserAudit.Rows[i + 1][TABLE.VbecsUser.UserName].ToString())
                    {
                        RowModified = false;
                    }
                    else
                    {
                        RowModified = true;
                    }
                }

                foreach (DataColumn dtCol in dtUserAudit.Columns)
                {
                    if (dtCol.ColumnName == TABLE.VbecsUser.UserDUZ ||
                        dtCol.ColumnName == TABLE.VbecsUser.VistaUserName ||
                        dtCol.ColumnName == TABLE.VbecsUser.UserEmail ||
                        dtCol.ColumnName == TABLE.VbecsUser.UserInitials ||
                        dtCol.ColumnName == TABLE.VbecsUser.RecordStatusCode)
                    {
                        //skip columns that have not changed the value
                        if (RowModified == true)
                        {
                            if (dtUserAudit.Rows[i][dtCol].Equals(dtUserAudit.Rows[i + 1][dtCol]))
                            {
                                continue;
                            }
                        }
                        else
                        {
                            if (dtUserAudit.Rows[i][dtCol].ToString().Trim() == "")
                            {
                                continue;
                            }
                        }

                        DataRow drAuditReport = this._dtAuditReport.NewRow();
                        drAuditReport[ARTIFICIAL.AuditType] = AuditTrailConst.AuditTrailType.MaintainUser;
                        drAuditReport[ARTIFICIAL.AuditIdentifier] = dtUserAudit.Rows[i][TABLE.VbecsUser.UserName].ToString().Trim() + " (" + dtUserAudit.Rows[i][TABLE.VbecsUser.VbecsUserId].ToString().Trim() + ")";

                        if (dtCol.ColumnName == TABLE.VbecsUser.UserDUZ)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "VistA DUZ";
                        }
                        else if (dtCol.ColumnName == TABLE.VbecsUser.VistaUserName)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "VistA user name";
                        }
                        else if (dtCol.ColumnName == TABLE.VbecsUser.UserEmail)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Email address";
                        }
                        else if (dtCol.ColumnName == TABLE.VbecsUser.UserInitials)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "User initials";
                        }
                        else if (dtCol.ColumnName == TABLE.VbecsUser.RecordStatusCode)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Active VBECS user indicator";
                        }

                        if (RowModified == false)
                        {
                            drAuditReport[ARTIFICIAL.OldValue] = "None";
                        }
                        else
                        {
                            if (dtCol.ColumnName == TABLE.VbecsUser.RecordStatusCode)
                            {
                                if (dtUserAudit.Rows[i + 1][dtCol].ToString().Trim() == Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active).ToString().Trim())
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "Active";
                                }
                                else
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "Inactive";
                                }
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.OldValue] = dtUserAudit.Rows[i + 1][dtCol].ToString().Trim();
                            }
                        }

                        if (dtCol.ColumnName == TABLE.VbecsUserDivisionRole.RecordStatusCode)
                        {
                            if (dtUserAudit.Rows[i][dtCol].ToString().Trim() == Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active).ToString().Trim())
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "Active";
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "Inactive";
                            }
                        }
                        else
                        {
                            drAuditReport[ARTIFICIAL.NewValue] = dtUserAudit.Rows[i][dtCol].ToString().Trim();
                        }

                        try
                        {
                            BOL.VbecsUser LastUpdateUser = new BOL.VbecsUser(dtUserAudit.Rows[i][TABLE.BloodUnitStatus.LastUpdateUser].ToString().Trim());
                            if (LastUpdateUser.UserName == null)
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = dtUserAudit.Rows[i][TABLE.BloodUnitStatus.LastUpdateUser];
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = LastUpdateUser.UserName;
                            }
                        }
                        catch
                        {
                            drAuditReport[ARTIFICIAL.AuditUser] = dtUserAudit.Rows[i][TABLE.BloodUnitStatus.LastUpdateUser];
                        }

                        drAuditReport[ARTIFICIAL.AuditDate] = dtUserAudit.Rows[i][ARTIFICIAL.MirrorDate];

                        drAuditReport[ARTIFICIAL.Comment] = "";

                        this._dtAuditReport.Rows.Add(drAuditReport);
                    }
                }
            }
        }

        /// <summary>
        /// Generates the DataTable used as a source for Interface Control Audit Report
        /// </summary>
        private void GetInterfaceControlAuditReport()
        {
            bool RowModified = false;
            DataTable dtInterfaceControlAudit = DAL.AuditTrail.GetInterfaceControlAudit(this.StartDate, this.EndDate);
            int nCount = 0;

            //generating one audit report row for each changed field
            nCount = dtInterfaceControlAudit.Rows.Count;
            for (int i = 0; i < nCount; i++)
            {
                if ((DateTime)dtInterfaceControlAudit.Rows[i][ARTIFICIAL.MirrorDate] < this.StartDate)
                    continue;
                if (i == dtInterfaceControlAudit.Rows.Count - 1)
                {
                    RowModified = false;
                }
                else
                {
                    //do not compare records of two different blood units
                    //if the record has just been added and there is no history of changes
                    if (dtInterfaceControlAudit.Rows[i][TABLE.InterfaceControl.InterfaceControlId].ToString() != dtInterfaceControlAudit.Rows[i + 1][TABLE.InterfaceControl.InterfaceControlId].ToString())
                    {
                        RowModified = false;
                    }
                    else
                    {
                        RowModified = true;
                    }
                }

                foreach (DataColumn dtCol in dtInterfaceControlAudit.Columns)
                {
                    if (dtCol.ColumnName == TABLE.InterfaceControl.InterfaceFacilityId ||
                        dtCol.ColumnName == TABLE.InterfaceControl.InterfaceIPAddress ||
                        dtCol.ColumnName == TABLE.InterfaceControl.InterfacePortNumber ||
                        dtCol.ColumnName == TABLE.InterfaceControl.VbecsIPAddress ||
                        dtCol.ColumnName == TABLE.InterfaceControl.VbecsPortNumber ||
                        dtCol.ColumnName == TABLE.InterfaceControl.VbecsFacilityId ||
                        dtCol.ColumnName == TABLE.InterfaceControl.PurgeSuccessfulMessage ||
                        dtCol.ColumnName == TABLE.InterfaceControl.PurgeUnSuccessfulMessage ||
                        dtCol.ColumnName == TABLE.InterfaceControl.LogEventsIndicator ||
                        dtCol.ColumnName == TABLE.InterfaceControl.AckTimeout ||
                        dtCol.ColumnName == TABLE.InterfaceControl.ReTransmitAttempts ||
                        dtCol.ColumnName == TABLE.InterfaceControl.InterfaceAdministratorName ||
                        dtCol.ColumnName == TABLE.InterfaceControl.InterfaceDomainName ||
                        // CR 2943
                        dtCol.ColumnName == TABLE.InterfaceControl.InterfaceActiveIndicator)
                    {
                        //skip columns that have not changed the value
                        if (RowModified == true)
                        {
                            if (dtInterfaceControlAudit.Rows[i][dtCol].Equals(dtInterfaceControlAudit.Rows[i + 1][dtCol]))
                            {
                                continue;
                            }
                        }
                        else
                        {
                            if (dtInterfaceControlAudit.Rows[i].IsNull(dtCol))
                                continue;
                        }

                        DataRow drAuditReport = this._dtAuditReport.NewRow();
                        drAuditReport[ARTIFICIAL.AuditType] = AuditTrailConst.AuditTrailType.InterfaceControl;

                        //CR 2943 - removing "_" from interface name, CR 2516 - InterfaceControl table split out into a new table, causing AutoGen file changes
                        drAuditReport[ARTIFICIAL.AuditIdentifier] = dtInterfaceControlAudit.Rows[i][TABLE.Interface.InterfaceName].ToString().Trim().Replace("_", " "); ;

                        if (dtCol.ColumnName == TABLE.InterfaceControl.InterfaceFacilityId)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Interface Facility ID";
                        }
                        else if (dtCol.ColumnName == TABLE.InterfaceControl.InterfaceIPAddress)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Interface IP Address";
                        }
                        else if (dtCol.ColumnName == TABLE.InterfaceControl.InterfacePortNumber)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Interface Port Number";
                        }
                        else if (dtCol.ColumnName == TABLE.InterfaceControl.InterfaceDomainName)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Interface Domain Name";
                        }
                        else if (dtCol.ColumnName == TABLE.InterfaceControl.VbecsIPAddress)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "VBECS IP Address";
                        }
                        else if (dtCol.ColumnName == TABLE.InterfaceControl.VbecsPortNumber)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "VBECS Port Number";
                        }
                        else if (dtCol.ColumnName == TABLE.InterfaceControl.VbecsFacilityId)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "VBECS Facility ID";
                        }
                        else if (dtCol.ColumnName == TABLE.InterfaceControl.PurgeSuccessfulMessage)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Completed Messages";
                        }
                        else if (dtCol.ColumnName == TABLE.InterfaceControl.PurgeUnSuccessfulMessage)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Messages In Error";
                        }
                        else if (dtCol.ColumnName == TABLE.InterfaceControl.LogEventsIndicator)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Log events and HL7 messages to Event Log indicator";
                        }
                        else if (dtCol.ColumnName == TABLE.InterfaceControl.AckTimeout)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Ack Timeout";
                        }
                        else if (dtCol.ColumnName == TABLE.InterfaceControl.ReTransmitAttempts)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Retransmit Attempts";
                        }
                        else if (dtCol.ColumnName == TABLE.InterfaceControl.InterfaceAdministratorName)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Interface Administrator Email";
                        }//CR 2943
                        else if (dtCol.ColumnName == TABLE.InterfaceControl.InterfaceActiveIndicator)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Interface Status";
                        }

                        if (RowModified == false)
                        {
                            drAuditReport[ARTIFICIAL.OldValue] = "None";
                        }
                        else
                        {
                            if (dtCol.ColumnName == TABLE.InterfaceControl.LogEventsIndicator)
                            {
                                if ((bool)dtInterfaceControlAudit.Rows[i + 1][dtCol] == true)
                                    drAuditReport[ARTIFICIAL.OldValue] = "Yes";
                                else
                                    drAuditReport[ARTIFICIAL.OldValue] = "No";
                            }
                            // CR 2943
                            else if (dtCol.ColumnName == TABLE.InterfaceControl.InterfaceActiveIndicator)
                            {
                                //CR 2947
                                if (dtInterfaceControlAudit.Rows[i + 1].IsNull(dtCol))
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = "None";
                                }
                                else
                                {
                                    if ((bool)dtInterfaceControlAudit.Rows[i + 1][dtCol] == true)
                                        drAuditReport[ARTIFICIAL.OldValue] = "Enabled";
                                    else
                                        drAuditReport[ARTIFICIAL.OldValue] = "Disabled";
                                }
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.OldValue] = dtInterfaceControlAudit.Rows[i + 1][dtCol].ToString().Trim();
                            }
                        }

                        if (dtCol.ColumnName == TABLE.InterfaceControl.LogEventsIndicator)
                        {
                            if ((bool)dtInterfaceControlAudit.Rows[i][dtCol] == true)
                                drAuditReport[ARTIFICIAL.NewValue] = "Yes";
                            else
                                drAuditReport[ARTIFICIAL.NewValue] = "No";
                        }
                        // CR 2943
                        else if (dtCol.ColumnName == TABLE.InterfaceControl.InterfaceActiveIndicator)
                        {
                            //CR 2947
                            if (dtInterfaceControlAudit.Rows[i].IsNull(dtCol))
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "None";
                            }
                            else
                            {
                                if ((bool)dtInterfaceControlAudit.Rows[i][dtCol] == true)
                                    drAuditReport[ARTIFICIAL.NewValue] = "Enabled";
                                else
                                    drAuditReport[ARTIFICIAL.NewValue] = "Disabled";
                            }
                        }
                        else
                        {
                            drAuditReport[ARTIFICIAL.NewValue] = dtInterfaceControlAudit.Rows[i][dtCol].ToString().Trim();
                        }

                        try
                        {
                            BOL.VbecsUser LastUpdateUser = new BOL.VbecsUser(dtInterfaceControlAudit.Rows[i][TABLE.BloodUnitStatus.LastUpdateUser].ToString().Trim());
                            if (LastUpdateUser.UserName == null)
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = dtInterfaceControlAudit.Rows[i][TABLE.InterfaceControl.LastUpdateUser];
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = LastUpdateUser.UserName;
                            }
                        }
                        catch
                        {
                            drAuditReport[ARTIFICIAL.AuditUser] = dtInterfaceControlAudit.Rows[i][TABLE.InterfaceControl.LastUpdateUser];
                        }

                        drAuditReport[ARTIFICIAL.AuditDate] = dtInterfaceControlAudit.Rows[i][ARTIFICIAL.MirrorDate];

                        drAuditReport[ARTIFICIAL.Comment] = "";

                        this._dtAuditReport.Rows.Add(drAuditReport);
                    }
                }
            }
        }

        /// <summary>
        /// Generates the DataTable used as a source for VistALink Control Audit Report
        /// </summary>
        private void GetVistALinkControlAuditReport()
        {
            bool RowModified = false;
            DataTable dtVistALinkControlAudit = DAL.AuditTrail.GetVistALinkControlAudit(this.StartDate, this.EndDate);
            int nCount = 0;
            //generating one audit report row for each changed field
            nCount = dtVistALinkControlAudit.Rows.Count;
            for (int i = 0; i < nCount; i++)
            {
                if ((DateTime)dtVistALinkControlAudit.Rows[i][ARTIFICIAL.MirrorDate] < this.StartDate)
                    continue;
                if (i == dtVistALinkControlAudit.Rows.Count - 1)
                {
                    RowModified = false;
                }
                else
                {
                    if (dtVistALinkControlAudit.Rows[i][TABLE.VistaLinkControl.ServerAppHandle].ToString() != dtVistALinkControlAudit.Rows[i + 1][TABLE.VistaLinkControl.ServerAppHandle].ToString())
                    {
                        RowModified = false;
                    }
                    else
                    {
                        RowModified = true;
                    }
                }

                foreach (DataColumn dtCol in dtVistALinkControlAudit.Columns)
                {
                    if (dtCol.ColumnName == TABLE.VistaLinkControl.VistaIpAddress ||
                        dtCol.ColumnName == TABLE.VistaLinkControl.VistaPortNumber ||
                        dtCol.ColumnName == TABLE.VistaLinkControl.VistaDomainName ||
                        dtCol.ColumnName == TABLE.VistaLinkControl.VbecsServiceIpAddress || //CR 3419
                        dtCol.ColumnName == TABLE.VistaLinkControl.VbecsServicePortNumber)  //CR 3419)
                    {
                        //skip columns that have not changed the value
                        if (RowModified == true)
                        {
                            if (dtVistALinkControlAudit.Rows[i][dtCol].Equals(dtVistALinkControlAudit.Rows[i + 1][dtCol]))
                            {
                                continue;
                            }
                        }
                        else
                        {
                            if (dtVistALinkControlAudit.Rows[i].IsNull(dtCol))
                                continue;
                        }

                        DataRow drAuditReport = this._dtAuditReport.NewRow();
                        drAuditReport[ARTIFICIAL.AuditType] = AuditTrailConst.AuditTrailType.InterfaceControl;
                        drAuditReport[ARTIFICIAL.AuditIdentifier] = "VistALink";

                        if (dtCol.ColumnName == TABLE.VistaLinkControl.VistaIpAddress)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "VistA TCP/IP Address";
                        }
                        else if (dtCol.ColumnName == TABLE.VistaLinkControl.VistaPortNumber)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "VistA TCP Port Number";
                        }
                        else if (dtCol.ColumnName == TABLE.VistaLinkControl.VistaDomainName)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "VistA Domain Name";
                        }
                        // CR 3419 start
                        else if (dtCol.ColumnName == TABLE.VistaLinkControl.VbecsServiceIpAddress)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "VBECS VistALink Service TCP/IP Address";
                        }
                        else if (dtCol.ColumnName == TABLE.VistaLinkControl.VbecsServicePortNumber)
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "VBECS VistALink Service TCP Port Number";
                        }
                        // CR 3419 end

                        if (RowModified == false)
                        {
                            drAuditReport[ARTIFICIAL.OldValue] = "None";
                        }
                        else
                        {
                            drAuditReport[ARTIFICIAL.OldValue] = dtVistALinkControlAudit.Rows[i + 1][dtCol].ToString().Trim();
                        }

                        drAuditReport[ARTIFICIAL.NewValue] = dtVistALinkControlAudit.Rows[i][dtCol].ToString().Trim();

                        try
                        {
                            BOL.VbecsUser LastUpdateUser = new BOL.VbecsUser(dtVistALinkControlAudit.Rows[i][TABLE.VistaLinkControl.LastUpdateUser].ToString().Trim());
                            if (LastUpdateUser.UserName == null)
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = dtVistALinkControlAudit.Rows[i][TABLE.VistaLinkControl.LastUpdateUser];
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = LastUpdateUser.UserName;
                            }
                        }
                        catch
                        {
                            drAuditReport[ARTIFICIAL.AuditUser] = dtVistALinkControlAudit.Rows[i][TABLE.VistaLinkControl.LastUpdateUser];
                        }

                        drAuditReport[ARTIFICIAL.AuditDate] = dtVistALinkControlAudit.Rows[i][ARTIFICIAL.MirrorDate];

                        drAuditReport[ARTIFICIAL.Comment] = "";

                        this._dtAuditReport.Rows.Add(drAuditReport);
                    }
                }
            }
        }

        /// <summary>
        /// Generates the DataTable used as a source for SpecimenTest Audit Report
        /// </summary>
        private void GetSpecimenTestAuditReport()
        {
            bool RowModified = false;
            DataTable dtSpecimenTestAudit = DAL.AuditTrail.GetSpecimenTestAudit(this.StartDate, this.EndDate);
            int nCount = 0;
            //generating one audit report row for each changed field
            nCount = dtSpecimenTestAudit.Rows.Count;
            for (int i = 0; i < nCount; i++)
            {
                if ((DateTime)dtSpecimenTestAudit.Rows[i][ARTIFICIAL.MirrorDate] < this.StartDate)
                    continue;
                if (i == dtSpecimenTestAudit.Rows.Count - 1)
                {
                    continue;
                }
                else
                {
                    //do not compare records of two different blood units
                    if (dtSpecimenTestAudit.Rows[i][TABLE.OrderedTest.OrderedTestGuid].ToString() != dtSpecimenTestAudit.Rows[i + 1][TABLE.OrderedTest.OrderedTestGuid].ToString() &&
                        (int)dtSpecimenTestAudit.Rows[i][ARTIFICIAL.MirrorId] != 0)
                    {
                        continue;
                    }
                    //if the record has just been added and there is no history of changes
                    else if (dtSpecimenTestAudit.Rows[i][TABLE.OrderedTest.OrderedTestGuid].ToString() != dtSpecimenTestAudit.Rows[i + 1][TABLE.OrderedTest.OrderedTestGuid].ToString() &&
                        (int)dtSpecimenTestAudit.Rows[i][ARTIFICIAL.MirrorId] == 0)
                    {
                        continue;
                    }
                    else
                    {
                        RowModified = true;
                    }
                }

                if (RowModified == false)
                    continue;

                foreach (DataColumn dtCol in dtSpecimenTestAudit.Columns)
                {
                    if (dtCol.ColumnName == TABLE.SpecimenTest.RecordStatusCode ||
                        dtCol.ColumnName == TABLE.SpecimenTest.AboRhChangeIndicator)
                    {
                        //skip columns that have not changed the value
                        if (RowModified == true)
                        {
                            if (dtSpecimenTestAudit.Rows[i][dtCol].Equals(dtSpecimenTestAudit.Rows[i + 1][dtCol]) ||
                                (dtSpecimenTestAudit.Rows[i].IsNull(dtCol) == true &&
                                dtSpecimenTestAudit.Rows[i + 1].IsNull(dtCol) == true) ||
                                ((int)dtSpecimenTestAudit.Rows[i][TABLE.SpecimenTest.LastUpdateFunctionId] != (int)Common.UpdateFunction.UC094FrmInvalidatePatientTestResults &&
                                (int)dtSpecimenTestAudit.Rows[i][TABLE.SpecimenTest.LastUpdateFunctionId] != (int)Common.UpdateFunction.UC100FrmJustifyAboRHChange))
                            {
                                continue;
                            }

                            if (dtCol.ColumnName == TABLE.SpecimenTest.RecordStatusCode)
                            {
                                if (dtSpecimenTestAudit.Rows[i + 1][TABLE.SpecimenTest.RecordStatusCode].ToString() == Convert.ToString(Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.ErrorCorrected)) ||
                                    dtSpecimenTestAudit.Rows[i + 1][TABLE.SpecimenTest.RecordStatusCode].ToString() == Convert.ToString(Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Inactive)))
                                {
                                    continue;
                                }
                            }
                            else
                            {
                                if ((bool)dtSpecimenTestAudit.Rows[i + 1][TABLE.SpecimenTest.AboRhChangeIndicator] == true)
                                {
                                    continue;
                                }
                            }
                        }

                        DataRow drAuditReport = this._dtAuditReport.NewRow();
                        drAuditReport[ARTIFICIAL.AuditType] = AuditTrailConst.AuditTrailType.SpecimenTest;

                        drAuditReport[ARTIFICIAL.AuditIdentifier] = dtSpecimenTestAudit.Rows[i][TABLE.Patient.PatientLastName].ToString().Trim() + ", ";
                        drAuditReport[ARTIFICIAL.AuditIdentifier] += dtSpecimenTestAudit.Rows[i][TABLE.Patient.PatientFirstName].ToString().Trim() + " (";
                        drAuditReport[ARTIFICIAL.AuditIdentifier] += dtSpecimenTestAudit.Rows[i][TABLE.Patient.DisplayVistaPatientId].ToString().Trim() + ")";

                        if (dtCol.ColumnName == TABLE.SpecimenTest.RecordStatusCode)
                        {
                            if (dtSpecimenTestAudit.Rows[i][TABLE.OrderableTest.OrderableTestName].ToString().Trim().IndexOf("TRW") == -1)
                            {
                                //all tests except TRW
                                drAuditReport[ARTIFICIAL.ChangedFieldName] = dtSpecimenTestAudit.Rows[i][TABLE.OrderableTest.OrderableTestName].ToString().Trim() + " (" + dtSpecimenTestAudit.Rows[i][TABLE.BloodTestType.BloodTestName].ToString().Trim() + ")";
                            }
                            else
                            {
                                //TRW - additional display of specimen to differentiate between pre and post specimens
                                drAuditReport[ARTIFICIAL.ChangedFieldName] = dtSpecimenTestAudit.Rows[i][TABLE.OrderableTest.OrderableTestName].ToString().Trim() + " (" + dtSpecimenTestAudit.Rows[i][TABLE.BloodTestType.BloodTestName].ToString().Trim() + ", Specimen UID: " + dtSpecimenTestAudit.Rows[i][TABLE.PatientSpecimen.SpecimenUid].ToString().Trim() + ")";
                            }
                            drAuditReport[ARTIFICIAL.Comment] = dtSpecimenTestAudit.Rows[i][TABLE.SpecimenTest.TestComments].ToString().Trim();
                        }
                        else
                        {
                            drAuditReport[ARTIFICIAL.ChangedFieldName] = "Patient ABO/Rh";
                            drAuditReport[ARTIFICIAL.Comment] = dtSpecimenTestAudit.Rows[i][TABLE.SpecimenTest.AboRhChangeJustification].ToString().Trim();
                        }


                        if (dtCol.ColumnName == TABLE.SpecimenTest.RecordStatusCode)
                        {
                            if (dtSpecimenTestAudit.Rows[i + 1][TABLE.SpecimenTest.RecordStatusCode].ToString() == Convert.ToString(Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.ErrorCorrected)))
                            {
                                drAuditReport[ARTIFICIAL.OldValue] = "Test Result Invalid";
                            }
                            else if (dtSpecimenTestAudit.Rows[i + 1][TABLE.SpecimenTest.RecordStatusCode].ToString() == Convert.ToString(Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Inactive)))
                            {
                                drAuditReport[ARTIFICIAL.OldValue] = "Invalid";
                            }
                            else
                            {
                                if (dtSpecimenTestAudit.Rows[i + 1][TABLE.BloodTestType.BloodTestName].ToString().Trim().IndexOf("Antibody ID") != -1)
                                {
                                    //for ABID results display Antibody Name not test result
                                    if (dtSpecimenTestAudit.Rows[i + 1][TABLE.AntibodyType.AntibodyTypeName].ToString().Trim() != "")
                                    {
                                        drAuditReport[ARTIFICIAL.OldValue] = dtSpecimenTestAudit.Rows[i + 1][TABLE.AntibodyType.AntibodyTypeName].ToString().Trim();
                                    }
                                    else
                                    {
                                        drAuditReport[ARTIFICIAL.OldValue] = "No Antibodies Identified";
                                    }
                                }
                                else
                                {
                                    drAuditReport[ARTIFICIAL.OldValue] = dtSpecimenTestAudit.Rows[i + 1][TABLE.BloodUnitTest.TestResultId].ToString().Trim();
                                }
                            }

                            if (dtSpecimenTestAudit.Rows[i][TABLE.SpecimenTest.RecordStatusCode].ToString() == Convert.ToString(Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.ErrorCorrected)))
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "Test Result Invalid";
                            }
                            else if (dtSpecimenTestAudit.Rows[i][TABLE.SpecimenTest.RecordStatusCode].ToString() == Convert.ToString(Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Inactive)))
                            {
                                drAuditReport[ARTIFICIAL.NewValue] = "Invalid";
                            }
                            else
                            {
                                if (dtSpecimenTestAudit.Rows[i][TABLE.BloodTestType.BloodTestName].ToString().Trim().IndexOf("Antibody ID") != -1)
                                {
                                    //for ABID results display Antibody Name not test result
                                    if (dtSpecimenTestAudit.Rows[i][TABLE.AntibodyType.AntibodyTypeName].ToString().Trim() != "")
                                    {
                                        drAuditReport[ARTIFICIAL.NewValue] = dtSpecimenTestAudit.Rows[i][TABLE.AntibodyType.AntibodyTypeName].ToString().Trim();
                                    }
                                    else
                                    {
                                        drAuditReport[ARTIFICIAL.NewValue] = "No Antibodies Identified";
                                    }
                                }
                                else
                                {
                                    drAuditReport[ARTIFICIAL.NewValue] = dtSpecimenTestAudit.Rows[i][TABLE.BloodUnitTest.TestResultId].ToString().Trim();
                                }
                            }
                        }
                        else
                        {
                            if ((int)dtSpecimenTestAudit.Rows[i + 1][TABLE.BloodUnitTest.BloodTestTypeId] == (int)Common.TestType.ABOInterp ||
                                (int)dtSpecimenTestAudit.Rows[i + 1][TABLE.BloodUnitTest.BloodTestTypeId] == (int)Common.TestType.ABOInterpTAS)
                            {
                                DataTable dtAboRhHistory = DAL.Patient.GetAboRHTestHistory((Guid)dtSpecimenTestAudit.Rows[i + 1][TABLE.Patient.PatientGuid], (DateTime)dtSpecimenTestAudit.Rows[i + 1][TABLE.SpecimenTest.TestDate]);
                                if (dtAboRhHistory.Rows.Count > 1)
                                {
                                    string AboRh = dtAboRhHistory.Rows[0][TABLE.SpecimenTest.TestResultId].ToString().Trim();
                                    if (dtAboRhHistory.Rows[1][TABLE.SpecimenTest.TestResultId].ToString().Trim() == "P")
                                    {
                                        AboRh += " Pos";
                                    }
                                    else if (dtAboRhHistory.Rows[1][TABLE.SpecimenTest.TestResultId].ToString().Trim() == "N")
                                    {
                                        AboRh += " Neg";
                                    }
                                    else if (dtAboRhHistory.Rows[1][TABLE.SpecimenTest.TestResultId].ToString().Trim() == "I")
                                    {
                                        AboRh += " Inc";
                                    }
                                    drAuditReport[ARTIFICIAL.OldValue] = AboRh;
                                }
                            }
                            else
                                continue;

                            if ((int)dtSpecimenTestAudit.Rows[i][TABLE.BloodUnitTest.BloodTestTypeId] == (int)Common.TestType.ABOInterp ||
                                (int)dtSpecimenTestAudit.Rows[i][TABLE.BloodUnitTest.BloodTestTypeId] == (int)Common.TestType.ABOInterpTAS)
                            {
                                string AboRh = dtSpecimenTestAudit.Rows[i][TABLE.BloodUnitTest.TestResultId].ToString().Trim();
                                DataRow[] drBloodUnitTestRh = dtSpecimenTestAudit.Select(TABLE.SpecimenTest.OrderedTestGuid + " = '" + dtSpecimenTestAudit.Rows[i][TABLE.SpecimenTest.OrderedTestGuid].ToString() + "' AND " + TABLE.SpecimenTest.AboRhChangeIndicator + " = " + dtSpecimenTestAudit.Rows[i][TABLE.SpecimenTest.AboRhChangeIndicator].ToString() + " AND (" + TABLE.SpecimenTest.BloodTestTypeId + " = " + (int)Common.TestType.RhInterp + " OR " + TABLE.SpecimenTest.BloodTestTypeId + " = " + (int)Common.TestType.RhInterpTAS + ")");
                                if (drBloodUnitTestRh.Length > 0)
                                {
                                    if (drBloodUnitTestRh[0][TABLE.BloodUnitTest.TestResultId].ToString().Trim() == "P")
                                    {
                                        AboRh += " Pos";
                                    }
                                    else if (drBloodUnitTestRh[0][TABLE.BloodUnitTest.TestResultId].ToString().Trim() == "N")
                                    {
                                        AboRh += " Neg";
                                    }
                                    else if (drBloodUnitTestRh[0][TABLE.BloodUnitTest.TestResultId].ToString().Trim() == "I")
                                    {
                                        AboRh += " Inc";
                                    }
                                }
                                drAuditReport[ARTIFICIAL.NewValue] = AboRh;
                            }
                            else
                                continue;
                        }

                        //if justified ABO/Rh is the same as the historic abo/rh we do not want to show the change
                        if (drAuditReport[ARTIFICIAL.OldValue].ToString().Trim() == drAuditReport[ARTIFICIAL.NewValue].ToString().Trim())
                            continue;

                        try
                        {
                            BOL.VbecsUser LastUpdateUser = new BOL.VbecsUser(dtSpecimenTestAudit.Rows[i][TABLE.SpecimenTest.LastUpdateUser].ToString().Trim());
                            if (LastUpdateUser.UserName == null)
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = dtSpecimenTestAudit.Rows[i][TABLE.SpecimenTest.LastUpdateUser];
                            }
                            else
                            {
                                drAuditReport[ARTIFICIAL.AuditUser] = LastUpdateUser.UserName;
                            }
                        }
                        catch
                        {
                            drAuditReport[ARTIFICIAL.AuditUser] = dtSpecimenTestAudit.Rows[i][TABLE.SpecimenTest.LastUpdateUser];
                        }

                        drAuditReport[ARTIFICIAL.AuditDate] = dtSpecimenTestAudit.Rows[i][ARTIFICIAL.MirrorDate];
                        this._dtAuditReport.Rows.Add(drAuditReport);
                    }
                }
            }
        }

        /// <summary>
        /// Sets up Audit Report DataTable
        /// </summary>
        /// <returns></returns>
        private DataTable CreateAuditReportTable()
        {
            DataTable dt = new DataTable("AuditReport");

            dt.Columns.Add(ARTIFICIAL.AuditIdentifier); // identifier of the audited record eg. Patient Name, Blood Unit ID, etc...
            dt.Columns.Add(ARTIFICIAL.AuditType); // type of audit - Patient, Blood Unit, QC, etc...
            dt.Columns.Add(ARTIFICIAL.ChangedFieldName);
            dt.Columns.Add(ARTIFICIAL.OldValue);
            dt.Columns.Add(ARTIFICIAL.NewValue);
            dt.Columns.Add(ARTIFICIAL.AuditUser);
            dt.Columns.Add(ARTIFICIAL.AuditDate, typeof(DateTime));
            dt.Columns.Add(ARTIFICIAL.Comment);

            dt.CaseSensitive = true;

            return dt;
        }

        ///<Developers>
        ///	<Developer>Krzysztof Dobranowski</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>3/24/2004</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="3998"> 
        ///		<ExpectedInput>None</ExpectedInput>
        ///		<ExpectedOutput>DataSet containing audited data</ExpectedOutput>
        ///	</Case>
        ///
        ///<Case type="1" testid ="5063"> 
        ///		<ExpectedInput>NA</ExpectedInput>
        ///		<ExpectedOutput>NA</ExpectedOutput>
        ///	</Case>
        ///
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// Gets the DataSet used as a source for Audit Trail Report
        /// </summary>
        /// <returns></returns>
        public DataSet GetAuditTrailReport()
        {
            this.GetTRAuditReport();
            this.GetBloodUnitFinanceAuditReport();
            this.GetLocalSupplierAuditReport();
            this.GetBloodProductAuditReport();
            this.GetDivisionAuditReport();
            this.GetAssociatedInstitutionAuditReport();
            this.GetProductModificationParameterAuditReport();
            this.GetAntibodyAuditReport();
            this.GetWorkloadAuditReport();
            this.GetTransfusionEffectivenessAuditReport();
            this.GetTransfusionComplicationsAuditReport();
            this.GetMSBOSAuditReport();
            this.GetComponentClassAuditReport();
            this.GetDailyControlAuditReport();
            this.GetReagentTypeParameterAuditReport();
            this.GetUserAlertsAuditReport();
            this.GetBloodUnitStatusAuditReport();
            this.GetBloodUnitModificationAuditReport();
            this.GetBloodUnitAuditReport();
            this.GetBloodUnitMediaAuditReport();
            this.GetBloodUnitAntigenAuditReport();
            this.GetBloodUnitTestAuditReport();
            this.GetUserRoleAuditReport();
            this.GetInterfaceControlAuditReport();
            this.GetVistALinkControlAuditReport();
            this.GetSpecimenTestAuditReport();

            DataSet ds = new DataSet("AuditTrailReport");
            ds.Tables.Add(this._dtAuditReport);

            DataTable dtPreliminaryAudit = new DataTable("PreliminaryAuditReport");
            dtPreliminaryAudit.Columns.Add(ARTIFICIAL.PreliminaryReportIndicator, typeof(bool));
            DataRow drPreliminaryAudit = dtPreliminaryAudit.NewRow();
            if (this.EndDate.Date.Equals(BOL.VBECSDateTime.GetDivisionCurrentDateTime().Date))
            {
                drPreliminaryAudit[ARTIFICIAL.PreliminaryReportIndicator] = 1;
            }
            else
            {
                drPreliminaryAudit[ARTIFICIAL.PreliminaryReportIndicator] = 0;
            }
            dtPreliminaryAudit.Rows.Add(drPreliminaryAudit);
            ds.Tables.Add(dtPreliminaryAudit);

            return ds;
        }

        ///<Developers>
        ///	<Developer>Rob Heiberger</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>1/31/2005</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="6648"> 
        ///		<ExpectedInput>Valid Parameters</ExpectedInput>
        ///		<ExpectedOutput>NA</ExpectedOutput>
        ///	</Case>
        ///
        ///
        ///<Case type="1" testid ="6649"> 
        ///		<ExpectedInput>Invalid Parameters</ExpectedInput>
        ///		<ExpectedOutput>NA</ExpectedOutput>
        ///	</Case>
        ///
        ///
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        /// <summary>
        /// GenerateAndPrintReport
        /// </summary>
        protected override VbecsReportPrinter createReportPrinter()
        {
            DataSet dsAuditTrailReport = this.GetAuditTrailReport();
            BOL.VbecsUser user = new BOL.VbecsUser(this.PrintedBy);
            string reportCriteria = "Date Range: ";
            reportCriteria += Common.VBECSDateTime.FormatDateString(this.StartDate);
            reportCriteria += " - ";
            reportCriteria += Common.VBECSDateTime.FormatDateString(this.EndDate);

            DataSet dsFooter = BOL.VbecsReport.GetReportFooter(reportCriteria);
            DataSet dsHeader = BOL.VbecsReport.GetReportHeader(user.UserName);
            ReportDataSource reportSourceAudit = new ReportDataSource();
            if (dsAuditTrailReport.Tables.Count > 0)
            {
                reportSourceAudit.Name = "AuditReport";
                reportSourceAudit.Value = dsAuditTrailReport.Tables[0];
            }

            ReportDataSource reportSourcePreliminary = new ReportDataSource();
            if (dsAuditTrailReport.Tables.Count > 1)
            {
                reportSourcePreliminary.Name = "PreliminaryAuditReport";
                reportSourcePreliminary.Value = dsAuditTrailReport.Tables[1];
            }

            ReportDataSource reportSourceHeader = new ReportDataSource();
            if (dsHeader.Tables.Count > 0)
            {
                reportSourceHeader.Name = "ReportHeaderDataSet";
                reportSourceHeader.Value = dsHeader.Tables[0];
            }

            ReportDataSource reportSourceFooter = new ReportDataSource();
            if (dsFooter.Tables.Count > 0)
            {
                reportSourceFooter.Name = "ReportFooterDataSet";
                reportSourceFooter.Value = dsFooter.Tables[0];
            }

            Assembly assembly = Assembly.LoadFrom(Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location) + "\\Reports.dll");
            Stream stream = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.AuditTrail.AuditTrailReport.rdlc");
            LocalReport report = new LocalReport();
            report.DataSources.Add(reportSourceAudit);
            report.DataSources.Add(reportSourcePreliminary);
            report.DataSources.Add(reportSourceHeader);
            report.DataSources.Add(reportSourceFooter);
            report.LoadReportDefinition(stream);
            return new BOL.VbecsReportPrinter(this.PrinterName, true, report);
        }
    }
	#endregion

}
